<?php

namespace App\Http\Controllers\Terminations\HrCollection;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Services\Terminations\HrCollection\RetrievalStatisticsReportGraphService;
use App\Services\Terminations\HrCollection\RetrievalStatisticsReportService;
use App\Services\Terminations\HrCollection\RetrievalStatisticsReportWidgetService;
use Exception;

class RetrievalStatisticsReportController extends Controller
{
    protected $service;
    protected $retrievalStatisticsReportGraphService;
    protected $widgetService;

    /**
     * Constructor for the class.
     *
     * @param RetrievalStatisticsReportService $service The retrieval statistics report service.
     * @param RetrievalStatisticsReportGraphService $retrievalStatisticsReportGraphService The retrieval statistics report graph service.
     * @param RetrievalStatisticsReportWidgetService $widgetService The retrieval statistics report widget service.
     *
     * @return void
     */
    public function __construct(
        RetrievalStatisticsReportService $service,
        RetrievalStatisticsReportGraphService $retrievalStatisticsReportGraphService,
        RetrievalStatisticsReportWidgetService $widgetService
    ) {
        $this->service = $service;
        $this->retrievalStatisticsReportGraphService = $retrievalStatisticsReportGraphService;
        $this->widgetService = $widgetService;
    }

    /**
     * Retrieves the graph data and displays the retrieval statistics report graph.
     */
    public function index()
    {
        $data = $this->retrievalStatisticsReportGraphService->getGraphData();

        return view('terminations.terms-report.retrieval-statistics-report-graph', $data);
    }

    /**
     * View a single bar report.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function viewSingleBarReport()
    {
        $filters = $this->service->getUrlFilters();

        return view('terminations.terms-report.retrieval-statistics-detail-report', compact('filters'));
    }

    /**
     * Retrieves the data for the function.
     *
     * @return ReportJsonResponse The JSON response containing the report data.
     */
    public function data()
    {
        try {
            $filteredData = $this->service->data();
            $assets =  $filteredData['assets'];
            $data = [];

            if (!empty($assets)) {
                $data = $this->service->getReportData($assets, request('start'));
            }

            return new ReportJsonResponse(request()->input('draw'), $data, $filteredData['count']);
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage()], 500);
        }
    }

    /**
     * Retrieves the widget data.
     *
     * @throws Exception when an error occurs while retrieving the widget data.
     * @return mixed The widget data.
     */
    public function widgetData()
    {
        try {
            return $this->widgetService->getWidgetData();
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage()], 500);
        }
    }

    /**
     * Export the data to CSV format.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $dataCollection = collect();
        $assets = $this->service->exportData();
        $assets->chunk(5000, function ($assetChunks) use ($data, $dataCollection) {
            $dataCollection->push($this->service->getReportExportData($assetChunks, $data));
        });

        return exportToCsv($dataCollection->toArray());
    }
}
