<?php

namespace App\Http\Controllers\Terminations\HrCollection;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Services\Terminations\HrCollection\TermsCollectionReportGraphService;
use App\Services\Terminations\HrCollection\TermsCollectionReportService;
use Exception;
use Illuminate\Support\Facades\Log;

class CollectionStatisticsReportController extends Controller
{
    protected $service;
    protected $termsCollectionReportGraphService;

    /**
     * Constructs a new instance of the class.
     *
     * @param TermsCollectionReportService $service The service for terms collection report.
     * @param TermsCollectionReportGraphService $termsCollectionReportGraphService The service for terms collection report graph.
     */
    public function __construct(TermsCollectionReportService $service, TermsCollectionReportGraphService $termsCollectionReportGraphService)
    {
        $this->service = $service;
        $this->termsCollectionReportGraphService = $termsCollectionReportGraphService;
    }

    /**
     * Index function.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $data = $this->termsCollectionReportGraphService->generateGraphData();
        $months = $data['months'] ?? [];
        $lastYearGraphData = $data['lastYearData'] ?? [];
        $currentYearGraphData = $data['currentYearData'] ?? [];

        return view('terminations.terms-report.bar-terms-report', compact('months', 'currentYearGraphData', 'lastYearGraphData'));
    }

    /**
     * report page for selected bar in collection graph
     */
    public function viewSingleBarReport()
    {
        $filters = $this->service->getFilters();

        return view('terminations.terms-report.collection-detail-report', compact('filters'));
    }

    /**
     * generates collection report data
     */
    public function data()
    {
        $filteredData = $this->service->data();
        $assets = $filteredData['assets'];
        $data = [];

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, request('start'));
        }

        return new ReportJsonResponse(request()->input('draw'), $data, $filteredData['count']);
    }

    /**
     * export the report data to csv
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $dataCollection = collect();
        $assets = $this->service->exportData();
        $assets->chunk(5000, function ($assetChunks) use ($data, $dataCollection) {
            $dataCollection->push($this->service->getReportExportData($assetChunks, $data));
        });

        return exportToCsv($dataCollection->toArray());
    }

    /**
     * manual sync
     */
    public function manualSync()
    {
        try {
            $this->service->updateCollectionDataByFilters();

            return response()->json(['status' => 'success']);
        } catch (Exception $e) {
            Log::channel('daily')->info('Collection statistics report manual sync error: ' . $e->getMessage());
        }

        return response()->json(['status' => 'error']);
    }
}
