<?php

namespace App\Http\Controllers\SoftwareLicense;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Services\SoftwareLicense\LicenseUserService;
use App\Services\SoftwareLicense\NonTeqUserService;
use Carbon\Carbon;

/**
 * Software License User Controller
 */
class SoftwareLicenseUserController extends Controller
{

    /**
     * Initialize the controller with the LicenseUserService and NonTeqUserService.
     *
     * @param \App\Services\LicenseUserService $licenseUserService The service to manage license users
     * @param \App\Services\NonTeqUserService  $nonTeqUserService  The service to manage non-Teq users
     */
    public function __construct(protected LicenseUserService $licenseUserService, protected NonTeqUserService $nonTeqUserService)
    {
    }

    /**
     * Retrieve and format the data for license users.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the formatted license user data
     */
    public function list(Request $request)
    {
        $data = [];
        $userData   = $this->licenseUserService->getLicenseUsers();
        $totalData  = $userData->count();
        $licenseUsers   = $this->licenseUserService->getLicenseUsersByLimit($userData);

        if (!empty($licenseUsers)) {
            $data = $this->licenseUserService->getLicensesUserData($licenseUsers, request('start'), $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Export license user data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response containing the CSV file for download
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $userData = $this->licenseUserService->getLicenseUsers();

        $userData->chunk(5000, function ($userChunks) use ($data, $datas) {
            $datas->push($this->licenseUserService->getExportUserData($userChunks, $data));
        });

        return exportToCsv($datas->toArray());
    }

    /**
     * Add a user to a software license.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message
     */
    public function add(Request $request)
    {
        $this->validate(request(), [
            'user_id' => 'required'
        ]);

        $checkExists = $this->licenseUserService->checkLicenseUserExists($request->user_id, $request->software_license_id);

        if ($checkExists > 0) {
            return redirect()->route('software-assets.edit', $request->software_license_id)->with('license-user-popup-error', 'Selected user already added with the asset');
        }

        $data = $request->except(['_token']);
        $data['joined_on'] = Carbon::now();

        $response = $this->licenseUserService->createLicenseUser($data);

        if ($response) {
            return redirect()->route('software-assets.edit', $request->software_license_id)->with('message', 'Asset user added successfully.');
        } else {
            return redirect()->route('software-assets.edit', $request->software_license_id)->with('error', 'Opertaion failed');
        }
    }

    /**
     * Remove a user from a software license.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message
     */
    public function remove()
    {
        $response = $this->licenseUserService->removeUserFromLicense(request('remove-license-user-id'), request('remove-license-id'));

        if ($response) {
            return redirect()->route('software-assets.edit', request('remove-license-id'))->with('message', 'Asset user removed successfully.');
        } else {
            return redirect()->route('software-assets.edit', request('remove-license-id'))->with('error', 'Opertaion failed');
        }
    }

    /**
     * Retrieve and format the data for non-Teqtivity users.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the formatted non-Teqtivity user data
     */
    public function nonTeqtivityUsers(Request $request)
    {
        $data   = [];
        
        $licenseNonTeqUsers =  $this->nonTeqUserService->getNonTeqtivityUsers();
        $totalData          = $licenseNonTeqUsers->count();
        $licenseNonTeqUsers = $this->nonTeqUserService->getLicenseUsersByLimit($licenseNonTeqUsers);

        if (!empty($licenseNonTeqUsers)) {
            $data = $this->nonTeqUserService->getNonTeqtivityUsersData($licenseNonTeqUsers, request('start'), $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Export non-Teqtivity user data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response containing the CSV file for download
     */
    public function nonTeqUsersExport()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $userData = $this->nonTeqUserService->getNonTeqtivityUsers();

        $userData->chunk(5000, function ($userChunks) use ($data, $datas) {
            $datas->push($this->nonTeqUserService->getExportNonTeqUserData($userChunks, $data));
        });

        return exportToCsv($datas->toArray());
    }
}
