<?php

namespace App\Http\Controllers\SoftwareLicense;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\SoftwareLicense\SoftwareLicenseSubscription;
use App\Models\AssetHistory;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

/**
 * Software License subscription Controller
 */
class SoftwareLicenseSubscriptionController extends Controller
{

    protected $subscriptionModel;
    protected $assetHistoryModel;

    /**
     * Constructor for SoftwareLicense subscription Controller
     * 
     * @param object $subscriptionModel
     * @param object $assetHistoryModel
     */
    public function __construct(SoftwareLicenseSubscription $subscriptionModel, AssetHistory $assetHistoryModel)
    {
        $this->model = $subscriptionModel;
        $this->assetHistoryModel = $assetHistoryModel;
    }

    /**
     * Listing Page for the Software License Categories
     * 
     * @return view
     */
    public function index()
    {
        $subscriptions = $this->model->orderBy('duration')->orderBy('duration_type')->get();

        return view('software-license.subscription.index', compact('subscriptions'));
    }

    /**
     * Create software License subscription
     */
    public function store()
    {
        $this->validate(request(), [
            'duration' => 'required|numeric|min:0|not_in:0',
            'duration_type' => 'required',
        ]);

        $subscriptionData = $this->model->where(['duration' => request('duration'), 'duration_type' => request('duration_type')])->first();

        if (!$subscriptionData) {
            $name = request('duration') . ' ' . (request('duration') == 1 ? request('duration_type') : (request('duration_type') . 's'));
            $this->model->create(['duration' => request('duration'), 'duration_type' => request('duration_type'), 'name' => $name]);
            $description = __('license-history.SoftwareLicenseSubscriptionCreated', [
                'subscription' => request('duration') . ' ' . request('duration_type')
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'software_license_subscription_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            $this->assetHistoryModel->create($assetHistory);

            return redirect('/software-assets-subscription');
        } else {
            return redirect('/software-assets-subscription')->withErrors(['Subscription already exists']);
        }
    }

    /**
     * ajax call for the category actions 
     */
    public function ajax(Request $request)
    {
        if (request('action') == 'edit') {
            return $this->update($request->all());
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * update license subscription
     */
    public function update()
    {
        $this->validate(request(), [
            'subscription_id' => 'required',
            'duration' => 'required|numeric|min:0|not_in:0',
            'duration_type' => 'required',
        ]);
        $subscriptionExist = $this->model->where(['duration' => request('duration'), 'duration_type' => request('duration_type')])->where('id', '!=', request('subscription_id'))->first();

        if ($subscriptionExist) {
            return response()->json(['status' => 'error', 'message' => 'subscription already exists']);
        }

        $subscription = $this->model->findOrFail(request('subscription_id'));
        $oldName = $subscription->name;
        $name = request('duration') . ' ' . (request('duration') == 1 ? request('duration_type') : (request('duration_type') . 's'));
        $subscription->update(['duration' => request('duration'), 'duration_type' => request('duration_type'), 'name' => $name]);
        $description = '';

        if ($oldName != $name) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Name',
                'oldValue'    => $oldName,
                'newValue'  => $name,
            ]);
        }
        $description = __('license-history.SoftwareLicenseSubscriptionUpdated', [
            'description' => $description
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'software_license_subscription_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        $this->assetHistoryModel->create($assetHistory);

        return response()->json(['status' => 'success', 'message' => 'subscription updated successfully']);
    }

    /**
     * delete license subscription
     * @param array $data
     */
    public function destroy()
    {
        $subscription = $this->model->findOrFail(request('subscription_id'));

        if (count($subscription->licenses)) {
            return response()->json(['status' => 'error', 'message' => 'This subscription cannot be deleted as it has licenses associated with it']);
        }
        $name = $subscription->name;
        $subscription->delete();
        $description = __('license-history.SoftwareLicenseSubscriptionDeleted', [
            'subscription' => $name
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'software_license_subscription_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        $this->assetHistoryModel->create($assetHistory);

        return response()->json(['status' => 'success', 'message' => 'subscription deleted successfully']);
    }

    /**
     * ajax search license subscription
     */
    public function search()
    {
        $name = request('searchText');
        $subscriptions = $this->model;

        if ($name) {
            $subscriptions = $subscriptions->where('name', 'like',  '%' . $name . '%');
        }
        $subscriptions = $subscriptions->get();
        $view['subscriptions'] = view('software-license.subscription.list', compact('subscriptions'))->render();

        return response()->json($view);
    }
}
