<?php

namespace App\Http\Controllers\SoftwareLicense;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\SoftwareLicense\SoftwareLicenseManufacturerRequest;
use App\Models\SoftwareLicense\SoftwareLicenseManufacturer;
use App\Models\AssetHistory;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

/**
 * Software License Manufacturer Controller
 */
class SoftwareLicenseManufacturerController extends Controller
{

    /**
     * Initialize the controller with the SoftwareLicenseManufacturer and AssetHistory models.
     *
     * @param \App\Models\SoftwareLicenseManufacturer $manufacturerModel The model to manage software license manufacturers
     * @param \App\Models\AssetHistory                $assetHistoryModel The model to manage asset history
     */
    public function __construct(protected SoftwareLicenseManufacturer $manufacturerModel, protected AssetHistory $assetHistoryModel)
    {
    }

    /**
     * Display the index view for software license manufacturers.
     *
     * @return \Illuminate\View\View The view for the software license manufacturers index
     */
    public function index()
    {
        $manufacturers = $this->manufacturerModel->orderBy('name')->get();

        return view('software-license.manufacturer.index', compact('manufacturers'));
    }

    /**
     * Store a new software license manufacturer.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response to the software assets manufacturers page with a success or error message
     */
    public function store()
    {
        $this->validate(
            request(),
            [
                'name' => 'required|unique:software_license_manufacturers,name',
            ],
            [
                'name.required' => 'The manufacturer name is required',
                'name.unique' => 'Manufacturer already exists',
            ]
        );

        $name = request('name');
        $manufacturerData = $this->manufacturerModel->where('name', $name)->first();

        if (!$manufacturerData) {
            $this->manufacturerModel->create(['name' => $name]);
            $description = __('license-history.SoftwareLicenseManufacturerCreated', [
                'manufacturer' => $name
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'software_license_manufacturer_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            $this->assetHistoryModel->create($assetHistory);

            return redirect('/software-assets-manufacturers');
        } else {
            return redirect('/software-assets-manufacturers')->withErrors(['Manufacturer already exists']);
        }
    }

    /**
     * Handle AJAX requests for software license manufacturers.
     *
     * @param \App\Http\Requests\SoftwareLicenseManufacturerRequest $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse The JSON response indicating success or a redirect response from the called method
     */
    public function ajax(SoftwareLicenseManufacturerRequest $request)
    {
        if (request('action') == 'edit') {
            return $this->update($request);
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * Update a software license manufacturer.
     *
     * @param \App\Http\Requests\SoftwareLicenseManufacturerRequest $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success of the update operation
     */
    public function update(SoftwareLicenseManufacturerRequest $request)
    {
        $manufacturer = $this->manufacturerModel->findOrFail(request('manufacturer_id'));
        $oldName = $manufacturer->name;
        $manufacturer->update(['name' => $request->name]);
        $description = '';

        if ($oldName != request('name')) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Name',
                'oldValue'    => $oldName,
                'newValue'  => request('name'),
            ]);
        }

        $description = __('license-history.SoftwareLicenseManufacturerUpdated', [
            'description' => $description
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'software_license_manufacturer_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $this->assetHistoryModel->create($assetHistory);

        return response()->json(['status' => 'success', 'message' => 'Manufacturer updated successfully']);
    }

    /**
     * Delete a software license manufacturer.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success or failure of the delete operation
     */
    public function destroy()
    {
        $manufacturer = $this->manufacturerModel->findOrFail(request('manufacturer_id'));

        if (count($manufacturer->licenses)) {
            return response()->json(['status' => 'error', 'message' => 'This Manufacturer cannot be deleted as it has licenses associated with it']);
        }

        $name = $manufacturer->name;
        $manufacturer->delete();

        $description = __('license-history.SoftwareLicenseManufacturerDeleted', [
            'manufacturer' => $name
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'software_license_manufacturer_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $this->assetHistoryModel->create($assetHistory);

        return response()->json(['status' => 'success', 'message' => 'Manufacturer deleted successfully']);
    }

    /**
     * Search for software license manufacturers.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered view of the search results
     */
    public function search()
    {
        $name = request('searchText');
        $manufacturers = $this->manufacturerModel;

        if ($name) {
            $manufacturers = $manufacturers->where('name', 'like',  '%' . $name . '%');
        }

        $manufacturers = $manufacturers->get();
        $view['manufacturers'] = view('software-license.manufacturer.list', compact('manufacturers'))->render();

        return response()->json($view);
    }
}
