<?php

namespace App\Http\Controllers\SoftwareLicense;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\CsvFileUploadRequest;
use App\Services\SoftwareLicense\LicenseService;
use App\Http\Requests\SoftwareLicense\LicenseRequest;
use App\Services\SoftwareLicense\SoftwareAssetsBulkUpload;
use App\Services\SoftwareLicense\LicenseKeyService;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

/**
 * Software License Bulk Upload Controller
 */
class SoftwareLicenseBulkUploadController extends Controller
{

    /**
     * Constructor for the class.
     *
     * @param LicenseService           $licenseService          An instance of LicenseService used for managing licenses.
     * @param SoftwareAssetsBulkUpload $assetsBulkUploadService An instance of SoftwareAssetsBulkUpload used for bulk uploading software assets.
     * @param LicenseKeyService        $licenseKeyService       An instance of LicenseKeyService used for managing license keys.
     */
    public function __construct(
        protected LicenseService $licenseService,
        protected SoftwareAssetsBulkUpload $assetsBulkUploadService,
        protected LicenseKeyService $licenseKeyService
    ) {
    }


    /**
     * Handles the bulk upload of software assets from a CSV file.
     *
     * This method sets unlimited execution time and memory limit for
     * processing large files, retrieves the uploaded file, and stores it
     * in a designated directory. It then imports the data using the
     * SoftwareAssetsBulkUpload service and validates the fields. Any
     * errors encountered during validation are rendered and returned as
     * a JSON response. If there are no errors, the software assets data
     * is prepared for rendering and returned as a JSON response.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return \Illuminate\Http\JsonResponse
     *         A JSON response containing the rendered errors or data view.
     */
    public function bulkUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/software_assets_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);
        $view = [];
        $count = session('count_add') ? session('count_add') : 0;
        $data = $this->assetsBulkUploadService->importUploadData($path, $count);

        if (!empty($data['error'])) {
            $errors = $data['error'];
            $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();
            return response()->json($view);
        }
        $csvData = $data['csvData'];

        if ($csvData) {

            $errors = $this->assetsBulkUploadService->validateFields($csvData);
            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            if (!empty(array_filter($errors))) {
                $csvData = [];
            }
            $view['data'] = view('software-license.partials.upload-software-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * Store software assets with bulk upload
     *
     * @param object $request
     */
    public function store(Request $request)
    {
        $created = 0;

        if (!$request->has('software_assets_data')) {
            return redirect('software-assets')->with('error', 'Please Add Some Assets To Create');
        }

        $postData = $request->input('software_assets_data');

        foreach ($postData as $singleRecord) {
            $decodedSingleRecord = json_decode($singleRecord, true);
            $assetData = $this->validateSoftwareAssetRecord($decodedSingleRecord['assetData']);
            $keyData = $assetData;
            unset($assetData['users_allowed_for_key']);
            unset($assetData['license_key']);
            if ($assetData['license_key_type'] == 'no_license') {
                $keyData = $assetData;
            }

            if (!$assetData) continue; //skip invalid columns

            $assetData = $this->formatAssetData($assetData);
            DB::beginTransaction();
            try {
                $response = $this->licenseService->createLicense($assetData);
                if (isset($keyData['license_key'])  && $keyData['license_key']) {
                    $keys = explode(PHP_EOL, $keyData['license_key']);
                    if ($keyData['license_key_type'] == 'single_license') {
                        $keys = [explode(PHP_EOL, $keyData['license_key'])[0]];
                    }
                    $this->licenseKeyService->storeLicenseKey($keys, $response->id, $keyData['key_usage_type'], $keyData['users_allowed_for_key']);
                }
                DB::commit();
                $created++;
            } catch (Exception $e) {
                DB::rollback();
                Log::error($e->getMessage());
                // dd($e->getMessage());
                continue;
            }
        }
        if ($created != 0) {
            return redirect('software-assets')->with('message', $created . ' new ' . ($created > 1 ? 'assets' : 'asset') . ' added successfully.');
        }

        return redirect('software-assets')->with('error', 'No new assets created. Please check the uploaded file.');
    }

    /**
     * Validate Software Asset Record before insert
     *
     * @param array $assetData
     *
     * @return mixed
     */
    private function validateSoftwareAssetRecord($assetData)
    {
        $customValidatior = new LicenseRequest();
        $assetValidationRule = $customValidatior->rules();
        $assetValidationMessage = $customValidatior->messages();
        unset($assetData['count_add']);
        $validator = Validator::make($assetData, $assetValidationRule, $assetValidationMessage);

        if ($validator->fails()) {
            //  dd($assetData);
            // dd($validator->getMessageBag()->toArray());
            Log::error('Validation Failed. Creating the software asset with name ' . $assetData['name'] . 'failed.');

            return false;
        }

        return $assetData;
    }

    /**
     * format asset data before insert
     *
     * @param array $assetData
     *
     * @return array
     */
    private function formatAssetData($assetData)
    {
        $assetData['status'] = array_search($assetData['status'], config('software-license.licence_status')) ?? 0;
        $assetData['billing_cycle'] = array_search($assetData['billing_cycle'], config('software-license.billing_cycles')) ?? '';
        $assetData['license_type'] = array_search($assetData['license_type'], config('software-license.license_types')) ?? '';
        $assetData['auto_renewal'] = $assetData['license_type'] == 'perpetual' ? 'n/a' : $assetData['auto_renewal'];
        // $assetData['license_key_type'] = array_search($assetData['license_key_type'], config('software-license.license_key_types')) ?? '';
        // $assetData['key_usage_type'] = array_search($assetData['key_usage_type'], config('software-license.key_usage_types')) ?? '';
        // $assetData['type'] = array_search($assetData['type'], config('software-license.types')) ?? '';
        // $subscriptionLength = array_search($assetData['subscription_length'], config('software-license.subscription_length'));
        // $assetData['subscription_length'] = ($assetData['license_type'] == 'perpetual') ? '' : ($subscriptionLength ? $subscriptionLength : '');

        return $assetData;
    }
}
