<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\Vendor\VendorCategoryService;
use App\Services\AssetHistory;
use App\Http\Requests\VendorCategory\StoreVendorCategory;
use App\Http\Requests\VendorCategory\UpdateVendorCategory;
use App\Models\VendorCategory;
use App\Http\Responses\ReportJsonResponse;


class VendorCategoriesController extends Controller
{

    public function __construct(protected VendorCategoryService $service, protected AssetHistory $assetHistory)
    {
        $this->service = $service;
    }

    /**
     * Display the vendor categories view.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return view('settings.vendor-categories.categories');
    }

    /**
     * Retrieve and return vendor category data as a JSON response.
     *
     * This function fetches filtered vendor category data and formats it
     * into a JSON response suitable for DataTables. It includes the total 
     * number of records, the number of filtered records, and the data itself.
     *
     * @param Request $request The incoming request instance containing input parameters.
     * 
     * @return \Illuminate\Http\JsonResponse JSON response containing the vendor category data.
     */
    public function data(Request $request)
    {
        $filteredData = $this->service->data();
        $vendors = $filteredData['vendors'];
        $start = request('start');
        $data = [];

        if (!empty($vendors)) {
            $data = $this->service->getData($vendors, $start, $data);
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Store a new vendor category in the database.
     *
     * This function handles the creation of a new vendor category based on
     * the validated request data. It generates a slug for the category name,
     * creates the vendor category, logs the action in the asset history, and
     * triggers a bulk update event. Finally, it redirects to the vendor categories
     * page with a success message.
     *
     * @param StoreVendorCategory $request The incoming request containing the vendor category data.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the vendor categories listing with a success message.
     */
    public function store(StoreVendorCategory $request)
    {
        $data = $request->only(['name']);
        $data['slug'] = str_slug($data['name']);

        VendorCategory::create($data);

        $description = __('history.VendorCategoryCreated', [
            'category' => $request['name'],
        ]);

        $this->service->createAssetHistory('vendor_category_created', $description);

        return redirect(route('vendor-categories.index'))->with('message', 'Vendor category created successfully.');
    }

    /**
     * Update an existing vendor category in the database.
     *
     * This function handles the updating of an existing vendor category based on
     * the validated request data. It generates a slug for the category name,
     * updates the vendor category, logs the action in the asset history, and
     * triggers a bulk update event. Finally, it redirects to the vendor categories
     * page with a success message.
     *
     * @param UpdateVendorCategory $request The incoming request containing the vendor category data.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the vendor categories listing with a success message.
     */
    public function update(UpdateVendorCategory $request)
    {
        $vendor = VendorCategory::find($request->id);
        $oldName = $vendor->name;
        $data = $request->only(['name']);
        $data['slug'] = str_slug($data['name']);

        if ($oldName == $data['name']) {
            return redirect(route('vendor-categories.index'))->with('message', 'Vendor category created successfully.');
        }

        $vendor->update($data);

        $description = __('history.VendorCategoryUpdated', [
            'old_name' => $oldName,
            'new_name' => $data['name'],
        ]);

        $this->service->createAssetHistory('vendor_category_updated', $description);

        return redirect(route('vendor-categories.index'))->with('message', 'Vendor category created successfully.');
    }

    /**
     * Retrieves the vendor category details based on the provided ID.
     *
     * @param int $id The vendor category ID.
     *
     * @return \Illuminate\Http\JsonResponse The vendor category ID and name in JSON format.
     */
    public function edit($id)
    {
        $vendor = VendorCategory::find($id);

        $data['id'] = $vendor->id;
        $data['name'] = $vendor->name;

        return response()->json($data);
    }


    /**
     * Function to remove the vendor
     * @param UpdateVendor $request
     *
     * @return [type]
     */
    public function destroy($id)
    {
        $vendorCategory = VendorCategory::with('vendors')->findOrFail($id);

        if ($vendorCategory->vendors->count() > 0) {
            return response()->json('This vendor category is in use and cannot be deleted.');
        }

        $description = __('history.VendorCategoryDeleted', [
            'category' => $vendorCategory->name,
        ]);

        $this->service->createAssetHistory('vendor_category_deleted', $description);
        $vendorCategory->delete();

        return response()->json('Success');
    }
}
