<?php

namespace App\Http\Controllers\Settings\SlackNotifications;

use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Models\SlackNotificationCredential;
use App\Services\Integrations\Slack\SlackApiIntegration;
use App\Services\Slack\SlackService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SlackNotificationsController extends Controller
{

    public function __construct(protected SlackApiIntegration $slackApiService, protected SlackService $slackService) {}

    /**
     * Retrieves the API credential with the slug 'slack_api' and
     * renders the 'settings.slack-notifications.index' view with
     * the credential as a parameter.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $credential = ApiCredential::where('slug', 'slack_api')->first();

        $configurationData = false;

        if ($this->slackService->checkSlackConnection()) {
            $configurationData = true;
        }

        return view('settings.slack-notifications.index', compact('credential', 'configurationData'));
    }

    /**
     * Connect to the specified slack api.
     *
     * @throws Exception If an error occurs during the connection.
     * @return Redirect A redirect response.
     */
    public function connect()
    {
        $state = Str::random(32);
        session(['slack_oauth_state' => $state]);
        $notificationAppCredentials = ApiCredential::where('slug', 'teqtivity_slack_notification_app')->first();
        $authUrl = config('services.slack.authorize_url') . '?scope=' . config('services.slack.bot_scopes') . '&user_scope=' . config('services.slack.user_scopes') . '&redirect_uri=' . url('/slack-notifications-callback/') . '&client_id=' . $notificationAppCredentials->user_name . '&state=' . $state;

        return redirect()->away($authUrl);
    }

    /**
     * Callback function for handling Slack OAuth.
     *
     * @param Request $request the HTTP request object
     * @throws Exception if there is an error requesting the access token
     * @return RedirectResponse a redirect response to the Slack api credentials page
     */
    public function callback(Request $request)
    {
        // Validate state
        $expectedState = session('slack_oauth_state');
        $request->session()->forget('slack_oauth_state');
        $providedState = $request->query('state');

        if (!isset($expectedState) || !isset($providedState) || $expectedState != $providedState) {
            return redirect('/slack-notifications')->with('error', 'The provided auth state did not match the expected value.');
        }

        // Authorization code should be in the "code" query param
        $authCode = $request->query('code');

        try {
            // Make the token request
            $accessDetails = $this->slackApiService->getAccessToken($authCode);
            if (!$accessDetails) {
                return redirect('/slack-notifications')->with('error', 'Error requesting api token.');
            }
            $this->slackService->storeAccessToken($accessDetails);
        } catch (Exception $e) {
            Log::channel('daily')->error($e->getMessage());
            return redirect('/slack-notifications')->with('error', 'Error requesting api token.');
        }

        return redirect('/slack-notifications')->with('message', 'Connected successfully.');
    }

    /**
     * Resets the access token for a Slack API credential.
     *
     * @param int $id The ID of the Slack API credential.
     * @throws Exception If the Slack API credential is not found.
     * @return \Illuminate\Http\RedirectResponse The redirect response to the '/slack-notifications' page with a success message.
     */
    public function resetAccessToken($id)
    {
        try {
            ApiCredential::where('id', $id)->update([
                'user_name' => NULL,
                'key' => NULL,
                'password' => NULL,
            ]);

            return redirect('/slack-notifications')->with('message', 'Slack API credential reset successfully.');
        } catch (Exception) {
            return redirect('/slack-notifications')->with('error', 'Slack API credential reset failed.');
        }
    }
}
