<?php

namespace App\Http\Controllers\Settings\SlackNotifications;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\SlackNotificationCredential;
use App\Services\Integrations\Slack\SlackApiIntegration;
use App\Services\Slack\SlackChannelsDataService;
use App\Services\Slack\SlackService;
use Exception;
use Illuminate\Http\Request;

class SlackChannelsConfigController extends Controller
{

    public function __construct(
        protected SlackApiIntegration $slackApiService,
        protected SlackService $slackService,
        protected SlackChannelsDataService $slackChannelsDataService
    ) {}

    /**
     * Updates a Slack notification credential.
     *
     * @param Request $request The HTTP request object.
     * @param int $id The ID of the credential to update.
     * @throws ModelNotFoundException if the credential is not found.
     * @return \Illuminate\Http\RedirectResponse The redirect response.
     */
    public function edit($id)
    {
        if (!$this->slackService->checkSlackConnection()) {
            return redirect('/slack-notifications')->with('error', 'Please connect to Slack first.');
        }

        $credential = SlackNotificationCredential::findOrFail($id);

        return view('settings.slack-notifications.slack-channels-config-edit', ['credential' => $credential]);
    }

    /**
     * Searches for a Slack channel by its ID and returns the channel details.
     *
     * @param Request $request The HTTP request object containing the channel ID.
     * @return \Illuminate\Http\JsonResponse A JSON response containing the channel details or an error message.
     */
    public function searchChannels(Request $request)
    {
        $request->validate([
            'channel_id' => 'required'
        ]);

        // Fetch channel details using the service
        $channelDetails = $this->slackService->getChannelInfoById($request->channel_id);

        // Check if the channel details are found and return a proper JSON response
        if (isset($channelDetails['channel']) === false || empty($channelDetails['channel']) === true) {
            return response()->json([
                'success' => false,
                'message' => 'Channel not found'
            ]);
        }

        return response()->json([
            'success' => true,
            'channel_name' => $channelDetails['channel']['name'] ?? null,
            'channel_id' => $channelDetails['channel']['id'] ?? null
        ]);
    }

    /**
     * Updates a Slack notification credential.
     *
     * @param Request $request The HTTP request object.
     * @param int $id The ID of the credential to update.
     * @throws ModelNotFoundException if the credential is not found.
     * @return \Illuminate\Http\RedirectResponse The redirect response.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'channel_id' => 'required',
            'channel_name' => ['required', 'regex:/^[^#]/'],
        ], [
            'channel_name.regex' => 'Please do not add # at begining.',
        ]);

        $credential = SlackNotificationCredential::findOrFail($id);
        $credential->update([
            'channel_name' => $request->channel_name,
            'channel_id' => $request->channel_id
        ]);

        return redirect('/slack-notifications')->with('message', 'Updated credentials successfully.');
    }

    /**
     * Updates a Slack notification status.
     *
     * @param Request $request The HTTP request object.
     * @throws ModelNotFoundException if the credential is not found.
     * @return \Illuminate\Http\RedirectResponse The redirect response.
     */
    public function updateSlackNotificationsStatus(Request $request)
    {
        $request->validate([
            'id' => ['required'],
            'status' => ['required']
        ]);
        try {
            $credential = SlackNotificationCredential::findOrFail($request->id);
            $credential->update(['status' => $request->status]);

            return response()->json(['success' => true, 'message' => 'Status updated successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'error' => 'Operation failed.']);
        }
    }


    /**
     * Send a Test slack notification
     *
     * This method sends a test notification to the connectd slack channel of selected credential.
     * @param int $credentialId SlackNotificationCredential
     * @redirects /slack-notifications
     */
    public function testNotification($credentialId)
    {
        if (!$this->slackService->checkSlackConnection()) {
            return redirect('/slack-notifications')->with('error', 'Please connect to Slack first.');
        }

        try {
            $credential = SlackNotificationCredential::findOrFail($credentialId);
            $this->slackService->sendMessageToChannel($credential->slug, $credential->name . ' : test notification');

            return redirect('/slack-notifications')->with('message', 'Test notification sent successfully.');
        } catch (\Exception $e) {
            return redirect('/slack-notifications')->with('error', 'Operation failed.');
        }
    }

    /**
     * Fetches the Slack notification channels data for the datatable.
     *
     * @return ReportJsonResponse A JSON response containing the data and count.
     */
    public function data()
    {
        try {
            $filteredData = $this->slackChannelsDataService->data();
            $channels =  $filteredData['channels'];
            $data = [];

            if (!empty($channels)) {
                $data = $this->slackChannelsDataService->getReportData($channels, request('start'));
            }

            return new ReportJsonResponse(request()->input('draw'), $data, $filteredData['count']);
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage()], 500);
        }
    }
}
