<?php

namespace App\Http\Controllers\Settings\Integrations;

use App\Http\Controllers\Controller;
use App\Http\Requests\Settings\Zoom\ZoomHardwareCredentialsRequest;
use App\Http\Requests\Settings\Zoom\ZoomHardwareCredentialUpdateRequest;
use App\Services\ExportManagement\ExportManagementService;
use App\Services\Zoom\ZoomHardwareCredentialsService;
use App\Services\Zoom\ZoomHardwareDataDownloadService;
use Exception;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class ZoomHardwareCredentialsController extends Controller
{
    /**
     * Constructor for the class.
     *
     * @param ZoomHardwareCredentialsService $credentialsService The ZoomCredentialsService instance.
     * @param ZoomHardwareDataDownloadService $zoomDownloadService The ZoomDataDownloadService instance.
     * @param ExportManagementService $exportManagementService The ExportManagementService instance.
     *
     * @return void
     */
    public function __construct(
        protected ZoomHardwareCredentialsService $credentialsService,
        protected ZoomHardwareDataDownloadService $zoomDownloadService,
        protected ExportManagementService $exportManagementService
    ) {}

    /**
     * Show the credentials listing.
     *
     * @return View
     */
    public function index()
    {
        $status = 0;
        $credentials = $this->credentialsService->getApiCredentials();

        if (!$credentials) {
            return redirect()->route('zoom-hardware.create');
        }

        if ($this->credentialsService->checkApiConnection()) {
            $status = 1;
        }

        return view('settings.zoom-hardware.index', compact('credentials', 'status'));
    }

    /**
     * Show the form for create new credentials
     *
     * @return View
     */
    public function create()
    {
        if ($this->credentialsService->getApiCredentials()) {

            return redirect()->route('zoom-hardware.index');
        }

        return view('settings.zoom-hardware.create');
    }

    /**
     * Save the new credentials
     *
     * @param ZoomCredentialsRequest $request The request object.
     *
     * @return View
     */
    public function store(ZoomHardwareCredentialsRequest $request)
    {
        if ($this->credentialsService->getApiCredentials()) {
            return redirect()->route('zoom-hardware.index')->with('error', 'Zoom credentials already exist. ');
        }

        $response = $this->credentialsService->storeApiCredentials($request->all());

        return $response ? redirect(route('zoom-hardware.index'))->with('message', __('message.created')) :
            redirect(route('zoom-hardware.index'))->with('error', 'Something went wrong. Try again later');
    }

    /**
     * Show the form for edit the credential
     *
     * @param int $id Credentials iD
     *
     * @return View
     */
    public function edit($id)
    {
        $credential = $this->credentialsService->getApiCredentials();

        if (!$credential) {
            return redirect()->route('zoom-hardware.create');
        }

        return view('settings.zoom-hardware.edit', compact('credential'));
    }

    /**
     * Update the credentials
     *
     * @param ZoomHardwareCredentialUpdateRequest $request The request object.
     * @param int $id  Zoom Credential ID
     *
     * @return View
     */
    public function update(ZoomHardwareCredentialUpdateRequest $request, $id)
    {
        $response = $this->credentialsService->updateApiCredentials($request->all(), $id);

        return $response ? redirect(route('zoom-hardware.index'))->with('message', 'Credentials updated successfully') :
            redirect(route('zoom-hardware.index'))->with('error', 'Something went wrong. Try again later');
    }

    /**
     * Connect to the Zoom API with the existing credentials.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function connect()
    {
        $connectionStatus = $this->credentialsService->setConnection();

        return $connectionStatus ? redirect(route('zoom-hardware.index'))->with('message', 'Connected successfully')
            : redirect(route('zoom-hardware.index'))->with('error', 'Connection failed. Please check your credentials.');
    }

    /**
     * Downloads the Zoom rooms data as a JSON file.
     *
     * @return \Illuminate\Http\Response The response containing the downloaded file.
     */
    public function downloadZoomRoomsData()
    {
        if ($this->credentialsService->checkApiConnection() === false) {
            return response()->json(['status' => 'error', 'message' => 'Connection failed.']);
        }

        $this->exportManagementService->zoomApiDataExport('zoom_rooms');

        return response()->json(['status' => 'success', 'message' => getGeneratedReportLink('zoom_api_response')]);
    }

    /**
     * Downloads the Zoom rooms location data as a JSON file.
     *
     * @return \Illuminate\Http\Response The response containing the downloaded file.
     */
    public function downloadZoomRoomsLocations()
    {
        if ($this->credentialsService->checkApiConnection() === false) {
            return response()->json(['status' => 'error', 'message' => 'Connection failed.']);
        }

        $this->exportManagementService->zoomApiDataExport('zoom_locations');

        return response()->json(['status' => 'success', 'message' => getGeneratedReportLink('zoom_api_response')]);
    }

    /**
     * Resets the connection by clearing the tokens.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function resetConnection()
    {
        $credentials = $this->credentialsService->getApiCredentials();

        $this->credentialsService->clearTokens($credentials);

        return redirect(route('zoom-hardware.index'));
    }


    /**
     * Test the API connection
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function test()
    {
        if ($this->credentialsService->checkApiConnection()) {
            return response()->json(['status' => 'success', 'message' => 'Connection is successful.']);
        }

        return response()->json(['status' => 'error', 'message' => 'Failed to connect. Try again later.']);
    }

    /**
     * Initiates a manual sync for the Zoom API.
     *
     * This function calls the 'zoomDataSync:run' artisan command.
     *
     * @throws Exception if an error occurs during the synchronization process
     *
     * @return \Illuminate\Http\JsonResponse with a status and message
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        try {
            Artisan::call('zoomHardwareDataSync:run');
            $response = Artisan::output();

            if (strpos($response, 'sync completed successfully') === false) {
                return response()->json(array('status' => 'error', 'response' => $response, 'message' => 'Sync failed. Please check error logs for more details.'));
            }

            return response()->json(array('status' => 'success', 'message' => 'Sync completed successfully.'));
        } catch (Exception $e) {
            Log::channel('daily')->error('Sync failed with error:' . $e->getMessage());

            return response()->json(array('status' => 'error', 'message' => 'Something went wrong. Try again later.'));
        }
    }
}
