<?php

namespace App\Http\Controllers\Settings\DiscoveryTools;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Services\Integrations\JumpCloudIntegration;
use Exception;
use Illuminate\Support\Facades\Artisan;

class JumpCloudCredentialsController extends Controller
{
    public function __construct(protected JumpCloudIntegration $jumpCloudIntegration) {}

    /**
     * Home page of JumpCloud in setings menu
     */
    public function index()
    {
        $credentials = $this->jumpCloudIntegration->getJumpcloudCredentials();

        if ($credentials) {
            $status = $this->jumpCloudIntegration->checkApiConnection();

            return view('settings.discovery-tools.jumpcloud.index', compact('credentials', 'status'));
        }

        return redirect(route('jumpcloud.create'));
    }

    /**
     * Display the JumpCloud credentials create page if not already created.
     *
     * @throws No specific exceptions are thrown.
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function create()
    {
        $credential = ApiCredential::where('slug', 'jumpcloud')->first();
        if ($credential) {
            return redirect('jumpcloud/' . $credential->id . '/edit');
        }
        return view('settings.discovery-tools.jumpcloud.create');
    }

    /**
     * Save the JumpCloud credantials
     * @param Request $request
     * 
     * @return [type]
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'api_key'   => 'required',
        ]);

        ApiCredential::create([
            'api_name' => 'JumpCloud',
            'slug' => 'jumpcloud',
            'url' => config('services.jumpcloud.base_url'),
            'key' => $request->api_key,
        ]);

        return redirect('/jumpcloud');
    }

    /**
     * Show the edit page
     * @param mixed $id
     * 
     * @return [type]
     */
    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            return view('settings.discovery-tools.jumpcloud.edit', compact('credential'));
        } catch (Exception $e) {
            return redirect(route('jumpcloud.index'))->with('error', 'Something went wrong. Try again later.');
        }
    }

    /**
     * Update the JumpCloud credentials
     * @param Request $request
     * @param mixed $id
     * 
     * @return [type]
     */
    public function update(Request $request, $id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            $credential->update([
                'key' => $request->api_key ? $request->api_key : $credential->key,
            ]);

            return redirect(route('jumpcloud.index'))->with('message', 'Credentials updated successfully.');
        } catch (Exception $e) {
            return redirect(route('jumpcloud.index'))->with('error', 'Something went wrong. Try again later.');
        }
    }

    /**
     * Import JumpCloud devices using console command
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        $output = Artisan::call('jumpCloud:sync');

        if ($output) {
            return response()->json(array('status' => 'success', 'message' => 'JumpCloud manual sync finished.'));
        }

        return response()->json(array('status' => 'error', 'message' => 'Connection failed.'));
    }
}
