<?php

namespace App\Http\Controllers\Settings\CloudAssets;

use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Services\Integrations\CloudAssets\AwsIntegration;
use Exception;
use Illuminate\Support\Facades\Artisan;

class AwsCloudAssetsController extends Controller
{
    protected $apiService;

    /**
     * A constructor for the class.
     *
     * @param AwsIntegration $apiService The AWS integration service
     */
    public function __construct(AwsIntegration $apiService)
    {
        $this->apiService = $apiService;
    }
    /**
     * Load the index page.
     */
    public function index()
    {
        $credentials = ApiCredential::where('slug', 'aws_cloud_assets')->first();

        if ($credentials) {
            $status = $this->apiService->checkApiConnection();

            return view('settings.cloud-assets.aws.index', compact('credentials', 'status'));
        }

        return redirect()->route('aws-cloud-assets.create');
    }

    /**
     * Creates a new item.
     *
     * @return view
     */
    public function create()
    {
        $credential = ApiCredential::where('slug', 'aws_cloud_assets')->first();
        $regions = config('aws-cloud-assets.regions');

        if ($credential) {
            return redirect('aws-cloud-assets/' . $credential->id . '/edit');
        }

        return view('settings.cloud-assets.aws.create', compact('regions'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @throws Exception description of exception
     * @return RedirectResponse
     */
    public function store()
    {
        request()->validate([
            'access_key' => 'required',
            'secret_access_key' => 'required',
            'regions' => 'required',
        ]);

        try {
            ApiCredential::create([
                'api_name'      => 'AWS',
                'slug'          => 'aws_cloud_assets',
                'user_name'     => request()->access_key,
                'password'      => request()->secret_access_key,
                'custom_data'   => implode(",", request()->regions),
            ]);

            return redirect()->route('aws-cloud-assets.index')->with('message', __('message.created'));
        } catch (Exception $e) {
            return redirect(route('aws-cloud-assets.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Edit a specific item.
     *
     * @param datatype $id description
     * @throws Exception description of exception
     * @return View
     */
    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            $regions = config('aws-cloud-assets.regions');

            return view('settings.cloud-assets.aws.edit', compact('credential', 'regions'));
        } catch (Exception $e) {
            return redirect(route('aws-cloud-assets.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Update a credential by its ID.
     *
     * @param int $id The ID of the credential to update
     * @throws Exception When something goes wrong
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($id)
    {
        request()->validate([
            'access_key' => 'required',
            'regions' => 'required',
        ]);

        try {
            $credential = ApiCredential::findOrFail($id);
            $credential->update([
                'user_name' => request()->access_key,
                'password'  => (request()->secret_access_key) ?: $credential->password,
                'custom_data'   => implode(",", request()->regions),
            ]);

            return redirect()->route('aws-cloud-assets.index')->with('message', __('message.updated'));
        } catch (Exception $e) {
            return redirect(route('aws-cloud-assets.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Connect to the API
     *
     * @throws Exception description of exception
     * @return RedirectResponse
     */
    public function connect()
    {
        try {
            $connection = $this->apiService->checkApiConnection();

            if ($connection) {
                return redirect()->route('aws-cloud-assets.index')->with('message', 'Connected successfully');
            }

            return redirect()->route('aws-cloud-assets.index')->withError('Connection failed');
        } catch (Exception $e) {
            return redirect(route('aws-cloud-assets.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * process AWS Sync
     */
    public function manualSync()
    {
        try {
            $connection = $this->apiService->checkApiConnection();

            if ($connection) {
                Artisan::call('AwsInstanceSync:run');

                return response()->json(array('status' => 'success', 'message' => "AWS Instance Sync Done"));
            }

            return response()->json(array('status' => 'error', 'message' => 'Connection failed'));
        } catch (Exception $e) {
            return response()->json(array('status' => 'error', 'message' => 'Something went wrong. Try again later'));
        }
    }
}
?>