<?php

namespace App\Http\Controllers\Settings;

use Exception;
use Illuminate\Http\Request;
use App\Models\ApiCredential;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;

class AzureSsoController extends Controller
{
    /**
     * Display the Azure SSO configuration page.
     * Creates a new credential record if one doesn't exist.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        try {
            // Get or create the Azure SSO credential record with default values
            $credential = ApiCredential::firstOrCreate(
                ['slug' => 'azure_sso'],
                [
                    'api_name' => 'Azure SSO',
                    'integration_status' => 0 // Default to disabled
                ]
            );

            return view('settings.azure-sso.index', compact('credential'));
            
        } catch (Exception $e) {
            Log::error('Failed to load Azure SSO settings page', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->with('error', 'Failed to load Azure SSO settings. Please try again.');
        }
    }

    /**
     * Update the Azure SSO configuration.
     *
     * Validates the request data and updates the Azure SSO credentials
     * in the database. Clears the cached configuration and logs the update.
     * Redirects to the Azure SSO index page with a success or error message.
     *
     * @param \Illuminate\Http\Request $request The request containing the Azure SSO data.
     * @param \App\Models\ApiCredential $azureSso The Azure SSO credential to be updated.
     *
     * @return \Illuminate\Http\RedirectResponse Redirects to the Azure SSO index page with a message.
     */
    public function update(Request $request, ApiCredential $azureSso)
    {
        try {
            // Common validation rules.
            $rules = [
                'sign_out_url'  => 'required_if:integration_status,==,1',
                'sign_on_url'   => 'required_if:integration_status,==,1',
                'issuer_url'    => 'required_if:integration_status,==,1',
            ];

            // Additional rule only when creating new or when data is empty.
            if (empty($azureSso->data)) {
                $rules['signing_certificate'] = 'required_if:integration_status,==,1';
            }

            $request->validate($rules);

            $updateData = [
                'url'                   => $request->issuer_url,
                'user_name'             => $request->sign_on_url,
                'project'               => $request->sign_out_url,
                'integration_status'    => (int)$request->integration_status
            ];

            // Only update certificate if provided.
            if (request()->has('signing_certificate') && !empty(request()->signing_certificate)) {
                $updateData['data'] = $request->signing_certificate;
            }

            $azureSso->update($updateData);

            // Clear cached configuration.
            Cache::forget('azure_sso_config');

            Log::info('Azure SSO configuration updated successfully', [
                'credential_id' => $azureSso->id,
                'updated_by' => auth()->id() ?? 'system'
            ]);

            return redirect()->route('azure-sso.index')->with('message', __('message.updated'));
        } catch (Exception $e) {
            Log::error('Failed to update Azure SSO configuration', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'input_data' => $request->except(['signing_certificate']) // Exclude sensitive data
            ]);

            return redirect()->route('azure-sso.index')->with('error', 'Something went wrong while updating the configuration. Please try again.');
        }
    }
}