<?php

namespace App\Http\Controllers\Security\Crowdstrike\Reports;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use Illuminate\Http\Request;
use App\Services\Security\Crowdstrike\Reports\ActiveCveService;

class ActiveCveReportController extends Controller
{

    /**
     * Initialize the controller with the ActiveCveService.
     *
     * @param \App\Services\ActiveCveService $service The service to manage active CVEs
     */
    public function __construct(protected ActiveCveService $service) {}

    /**
     * Display the index view for active CVEs.
     *
     * @return \Illuminate\View\View The view for the active CVEs index
     */
    public function index()
    {
        return view('security.crowdstrike.active-cve.index');
    }

    /**
     * Retrieve and format the data for the active CVEs report.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \App\Http\Responses\ReportJsonResponse The JSON response containing the formatted report data
     */
    public function data(Request $request)
    {
        $filteredData = $this->service->data();

        if (!empty($filteredData['activeCveData'])) {
            $data = $this->service->getReportData($filteredData['activeCveData'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export the active CVEs report data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse The response to export the data as CSV
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $activeCveData = $this->service->exportData();

        $activeCveData->chunk(5000, function ($activeCveDataChunks) use ($data, $datas) {
            $datas->push($this->service->getReportExportData($activeCveDataChunks, $data));
        });

        return exportToCsv($datas->toArray());
    }
}
