<?php

namespace App\Http\Controllers\Security\Crowdstrike;

use App\Http\Controllers\Controller;
use App\Http\Requests\Security\Crowdstrike\CrowdstrikeCredentialsRequest;
use App\Services\Security\Crowdstrike\CrowdstrikeApiService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;

class CrowdstrikeCredentialsController extends Controller
{

    /**
     * Constructor for CrowdstrikeCredentialsController
     *
     * @param \App\Services\CrowdstrikeApiService $service The service to manage Crowdstrike API interactions
     */
    public function __construct(protected CrowdstrikeApiService $service)
    {
    }

    /**
     * If the user has already created the Software asset API credential, show the user the settings page.
     * Otherwise, redirect the user to the create page
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse The view for the Crowdstrike settings index or a redirect response to create credentials
     */
    public function index()
    {
        $credentialsData =  $this->service->getCredentials();

        if (!$credentialsData) {

            return redirect()->route('crowdstrike.create');
        }

        $status = $this->service->checkConnection($credentialsData);

        return view('settings.crowdstrike.index', ['credentials' => $credentialsData,'status' => $status]);
    }

    /**
     * Display the create view for Crowdstrike credentials.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse The view for creating Crowdstrike credentials or a redirect response to the index view
     */
    public function create()
    {
        if ($this->service->getCredentials()) {
            return redirect()->route('crowdstrike.index');
        }

        return view('settings.crowdstrike.create');
    }

    /**
     * Store the Crowdstrike credentials.
     *
     * @param \App\Http\Requests\CrowdstrikeCredentialsRequest $request The incoming request instance
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response to the index view with a success or error message
     */
    public function store(CrowdstrikeCredentialsRequest $request)
    {
        if ($this->service->getCredentials()) {
            return redirect()->route('crowdstrike.index')->with('error','Crowdstrike credentials already exist.');
        }

        $response = $this->service->addCredentials($request);

        return $response ? redirect()->route('crowdstrike.index')->with('message', __('message.created'))
            : redirect()->route('crowdstrike.create')->with('error', 'Operation Failed');
    }

    /**
     * Display the edit view for Crowdstrike credentials.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse The view for editing Crowdstrike credentials or a redirect response to the create view
     */
    public function edit()
    {
        $credentialsData = $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect()->route('crowdstrike.create');
        }

        return view('settings.crowdstrike.edit', ['credential' => $credentialsData]);
    }

    /**
     * Update the Crowdstrike credentials.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response to the index view with a success or error message
     */
    public function update(Request $request)
    {
        $response = $this->service->updateCredentials($request);

        return $response ? redirect(route('crowdstrike.index'))->with('message', 'Credentials updated successfully') :
            redirect(route('crowdstrike.index'))->with('error', 'Something went wrong. Try again later');
    }

    /**
     * Manually synchronize Crowdstrike data.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response to the index view with a success or error message
     */
    public function manualSync()
    {
        Artisan::call('Crowdstrike:sync');

        $softwareAssetSyncStatus = Artisan::output();
        $response = json_decode($softwareAssetSyncStatus, true);

        if ($response['status'] == 'success') {
            return redirect(route('crowdstrike.index'))->with( 'message', $response['message']);
        }

        return redirect(route('crowdstrike.index'))->with('error', $response['message']);
    }
}