<?php

namespace App\Http\Controllers\Revivn;

use App\Http\Controllers\Controller;
use App\Http\Requests\RevivnUpdateAssetRequest;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use App\Services\Revivn\RevivnAssetsManagementService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class RevivnAssetsManagementController extends Controller
{

    public function __construct(protected RevivnAssetsManagementService $service)
    {
    }

    /**
     * Get Asset Details
     *
     * @param int $id
     *
     * @return Array
     */
    public function getAsset($id)
    {
        $asset = Asset::find($id);

        $data['serial_no']   = $asset->serial_no;
        $data['hardware']    = $asset->make_and_model_id;
        $data['techspec']    = $asset->technical_spec_id;
        $data['assettype']   = $asset->asset_type_id;

        return $data;
    }

    /**
     * Get Hardware standard name and id
     *
     * @return Mixed
     */
    public function getHardware()
    {
        $hardwareStandards = MakeAndModel::where('asset_type_id', '=', request('assettype'))->get()->pluck('name', 'id');

        return $hardwareStandards ?? '';
    }

    /**
     * Get Technical specs details and id
     *
     * @return Mixed
     */
    public function getTechSpech()
    {
        $techSpecs = TechnicalSpecs::where('make_and_model_id', '=', request('techSpec'))->get()->pluck('details', 'id');

        return $techSpecs ?? '';
    }

    /**
     * Update the Single asset from modal
     **/
    public function updateAsset(RevivnUpdateAssetRequest $request)
    {
        $asset = Asset::with('assetType', 'makeAndModel', 'technicalSpec')->find(request('id'));
        if (!$asset) {
            return "operation failed.";
        }
        $data = $this->getDataToUpdate();
        $attributeUpdate = $this->createAttributeUpdateHistory($asset);
        if (!$attributeUpdate) {
            return response()->json("success");
        }
        $asset->update($data);

        return response()->json("success");
    }

    /**
     * Get the data to update.
     *
     * @return array
     */
    public function getDataToUpdate()
    {
        return [
            'make_and_model_id' => request('make_and_model_id'),
            'asset_type_id'     => request('asset_type_id'),
            'technical_spec_id' => request('technical_spec'),
        ];
    }

    /**
     * Create attribute update history for the given asset.
     *
     * @param datatype $asset description
     * @return Some_Return_Value
     */
    public function createAttributeUpdateHistory($asset)
    {
        $attributeUpdate = [];

        if ($asset->asset_type_id != request('asset_type_id')) {
            $attributeUpdate[] =  __('history.AttributeUpdated', [
                'attributeName' => 'asset type',
                'oldValue' => optional($asset->assetType)->name,
                'newValue' => optional(AssetType::find(request('asset_type_id')))->name,
            ]);
        }

        if ($asset->make_and_model_id != request('make_and_model_id')) {
            $attributeUpdate[] =  __('history.AttributeUpdated', [
                'attributeName' => 'hardware standard',
                'oldValue' => optional($asset->makeAndModel)->name,
                'newValue' => optional(MakeAndModel::find(request('make_and_model_id')))->name,
            ]);
        }

        if ($asset->technical_spec_id != request('technical_spec')) {
            $attributeUpdate[] =  __('history.AttributeUpdated', [
                'attributeName' => 'technical specs',
                'oldValue' => optional($asset->technicalSpec)->details,
                'newValue' => optional(TechnicalSpecs::find(request('technical_spec')))->details,
            ]);
        }

        // Array empty indicates no updates done.
        if (empty($attributeUpdate)) {
            return false;
        }
        $attributeUpdate = implode(', ', $attributeUpdate);

        $description = __('history.AssetAttributesUpdated', [
            'attributeUpdates' => $attributeUpdate,
            'assetName' => $asset->serial_no,
            'assetId' => $asset->id,
        ]);

        AssetHistory::create([
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'action' => 'asset_attributes_updated',
            'description' => $description
        ]);

        return true;
    }

    /**
     * Get the filter fields based on the request parameters.
     *
     * @param Request $request The request object
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFilterFields(Request $request)
    {
        if (!empty($request->search)) {

            $result = $this->service->getFilterFields($request->search, $request->type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }
}
