<?php

namespace App\Http\Controllers\Reports;

use App\Http\Controllers\Controller;
use App\Http\Traits\DataAndExportControllerTrait;
use App\Services\Reports\UsersToAssetsRatioService;
use Exception;
use Illuminate\Http\Request;

class UsersToAssetsRatioController extends Controller
{
    use DataAndExportControllerTrait;

    /**
     * Constructs a new instance of the class.
     *
     * @param UsersToAssetsRatioService $service the asset type to user service
     */
    public function __construct(protected UsersToAssetsRatioService $service) {}

    /**
     * Report page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $filterDataOptions = $this->service->getFilterDataOptions();

        return view('reports.users-to-asset-ratio.index', compact('filterDataOptions'));
    }

    /**
     * Show graphical represntation of the report
     * @param Request $request
     *
     * @return json
     */
    public function graphData()
    {
        $datas = $this->service->getGraphData();
        $chartHtml = view('reports.users-to-asset-ratio.partials.chart')->render();
        $graphData = $datas;
        $countLabels = ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12', '13', '14', '15+'];

        return json_encode(compact('chartHtml', 'graphData', 'countLabels'));
    }

    /**
     * Exports data to CSV.
     *
     * @return array The exported data in CSV format.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        try {
            $assets = $this->service->exportData();
            $assets->chunk(5000, function ($assetChunks) use ($data, $datas) {
                $datas->push($this->service->getReportExportData($assetChunks, $data));
            });

            return $datas->toArray();
        } catch (Exception $e) {
            return redirect()->route('users-to-asset-ratio')->with('error', 'Some error occurred. Please try again later');
        }
    }


    /**
     * Export the graph with list data.
     *
     * @throws Exception Some error occured. Please try again later
     * @return mixed
     */
    public function exportGraphWithListData()
    {
        try {
            setUnlimitedExecutionTimeAndMemoryLimit();

            $dataCollection = $this->export();
            $excel = $this->service->exportGraphAndDataListToExcel($dataCollection);

            return $excel;
        } catch (Exception $e) {
            return redirect()->route('users-to-asset-ratio')->with('error', 'Some error occured. Please try again later');
        }
    }
}
