<?php

namespace App\Http\Controllers\Reports;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use Illuminate\Http\Request;
use App\Models\AssetStatus;
use App\Services\Reports\OnHandAssetAgeService;

class OnHandAssetAgeController extends Controller
{
    protected $service;

    /**
     * Constructs a new instance of the class.
     *
     * @param OnHandAssetAgeService $service The OnHandAssetAgeService object.
     */
    public function __construct(OnHandAssetAgeService $service)
    {
        $this->service = $service;
    }

    /**
     * Report page view.
     *
     * @return view file
     */
    public function index()
    {
        $statuses = AssetStatus::whereIn('slug', ['brand_new', 'used', 'like_new'])->get();
        $ageFilterOptions = config('reports.on_hand_asset_age_options.age_filter_options');

        return view('reports.on-hand-asset-age.index', compact('statuses', 'ageFilterOptions'));
    }

    /**
     * Generates data for report
     *
     * @return object
     */
    public function data(Request $request)
    {
        $filteredData = $this->service->data();

        $assets =  $filteredData['assets'];
        $data = [];

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

     /**
     * Generates csv from report data
     *
     * @return object
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();
        $assets = $this->service->exportData();

        $assets->chunk(5000, function ($assetChunks) use ($data, $datas){
            $datas->push($this->service->getReportExportData($assetChunks, $data));
        });

        return exportToCsv($datas->toArray());
    }
}
