<?php

namespace App\Http\Controllers\Reports;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Services\Reports\DepreciationValue\DepreciationValueChartService;
use App\Services\Reports\DepreciationValue\DepreciationValueDataTableService;
use Exception;

class DepreciationValueController extends Controller
{
	protected $service;
	protected $depreciationValueDataTableService;

	/**
	 * Constructs a new instance of the class.
	 *
	 * @param DepreciationValueChartService $service The DepreciationValueChartService instance.
	 * @param DepreciationValueDataTableService $depreciationValueDataTableService The DepreciationValueDataTableService instance.
	 */
	public function __construct(DepreciationValueChartService $service, DepreciationValueDataTableService $depreciationValueDataTableService)
	{
		$this->service = $service;
		$this->depreciationValueDataTableService = $depreciationValueDataTableService;
	}

	/**
	 * Retrieves the data from the service and renders the 'reports.depreciation-value.index' view.
	 *
	 * @return \Illuminate\Contracts\View\View The rendered view.
	 */
	public function index()
	{
		$data = $this->service->data();

		return view('reports.depreciation-value.index', $data);
	}

	/**
	 * Retrieves the chart data and returns it as a JSON-encoded string.
	 *
	 * @return string JSON-encoded string containing the chart HTML, graph array, and years label.
	 */
	public function chart()
	{
		$data = $this->service->chartData();

		if (!$data) {
			return response()->json([
				'chartHtml' => '',
				'graphArray' => [],
				'yearsLabel' => [],
			]);
		}
		$chartHtml = view('reports.depreciation-value.partials.chart', $data)->render();
		$graphArray = $data['graphArray'];
		$yearsLabel = $data['yearsLabel'];

		return json_encode(compact('chartHtml', 'graphArray', 'yearsLabel'));
	}

	/**
	 * Retrieves the data for the API endpoint.
	 *
	 * @return ReportJsonResponse The JSON response containing the data for the API endpoint.
	 */
	public function data()
	{
		$filteredData = $this->depreciationValueDataTableService->data();
        $assets =  $filteredData['assets'];
        $data = [];

        if (!empty($assets)) {
            $data = $this->depreciationValueDataTableService->getReportData($assets, request('start'));
        }

        return new ReportJsonResponse(request()->input('draw'), $data, $filteredData['count']);
	}

	/**
	 * Export the graph with list data.
	 *
	 * @throws Exception Some error occured. Please try again later
	 * @return mixed
	 */
	public function exportGraphWithListData()
	{
		try {
        	setUnlimitedExecutionTimeAndMemoryLimit();
		
			$dataCollection = $this->export();
			$excel = $this->service->exportGraphAndDataListToExcel($dataCollection);

			return $excel;
		} catch (Exception $e) {
			return redirect()->route('depreciation-value')->with('error', 'Some error occured. Please try again later');
		}
	}

	/**
	 * Export the data.
	 *
	 * @return array
	 */
	public function export()
	{
        $data = [];
        $dataCollection = collect();
        $assets = $this->depreciationValueDataTableService->exportData();
        $assets->chunk(5000, function ($assetChunks) use ($data, $dataCollection) {
            $dataCollection->push($this->depreciationValueDataTableService->getReportExportData($assetChunks, $data));
        });

        return $dataCollection->toArray();
	}
}
