<?php

namespace App\Http\Controllers\Reports;

use App\Http\Controllers\Controller;
use App\Http\Traits\DataAndExportControllerTrait;
use App\Models\AssetType;
use App\Services\Reports\DamagedTrendsService;
use Exception;
use Illuminate\Support\Facades\Log;

/**
 *  DamagedTrendsController
 *
 * Controller for the DamagedTrends report
 * Other the Laptop asset type hardware will be validated. For laptops hardware standard will take values with asset type.
 */
class DamagedTrendsController extends Controller
{
    use DataAndExportControllerTrait;

    /**
     * Constructor for the Damaged Trend Controller
     *
     * @param DamagedTrendsService
     */
    public function __construct(protected DamagedTrendsService $service) {}

    /**
     * Report page View
     *
     * @return View
     */
    public function index()
    {
        $filterDataOptions = $this->service->getFilterDataOptions();
        $selectedData = [];

        return view('reports.damaged-trends.index', compact('filterDataOptions', 'selectedData'));
    }

    /**
     * A function to generate graph data based on the request parameters.
     *
     * @return view
     */
    public function graphData()
    {
        //validation
        $this->validate(
            request(),
            [
                'date-range' => 'required',
                'from-date' => 'required_if:date-range,custom',
                'to-date' => 'required_if:date-range,custom|date|nullable|after_or_equal:from-date',
            ],
            [
                'from-date.required_if' => 'From Date is required',
                'to-date.required_if' => 'From Date is required',
            ]
        );

        $laptopAssetTypeId = AssetType::where('slug', 'laptop')->first()->id;
        $isLaptopOnlyFilter = false;

        if (request()->has('asset_type')) {
            $isLaptopOnlyFilter = count(request('asset_type')) == 1 && request('asset_type')[0] == $laptopAssetTypeId && empty(request('make_and_model')) ? true : false;
        }

        if (!$isLaptopOnlyFilter && empty(request('make_and_model'))) {
            return redirect('report/damaged-trends')->withErrors([
                'make_and_model' => 'Hardware Standard is required',
            ])->withInput();
        }

        $filterDataOptions = $this->service->getFilterDataOptions();
        $graphData = $this->service->generateGraphData($isLaptopOnlyFilter);
        $months = $graphData['months'];
        $datas = $graphData['datas'];
        $selectedMakeAndModels = $graphData['selectedMakeAndModels'];
        $assetList = $graphData['assetList'];
        $monthNames = collect($months)->flatten()->toArray();
        $selectedData = $this->getSelectedData();

        return view('reports.damaged-trends.index', compact('filterDataOptions', 'datas', 'monthNames', 'selectedMakeAndModels', 'selectedData', 'assetList'));
    }

    /**
     * Retrieves selected data from the request.
     *
     * @return array selected data from the request
     */
    public function getSelectedData()
    {
        return [
            'type' => request('report_type'),
            'asset_type' => request('asset_type'),
            'location' => request('location'),
            'department' => request('department'),
            'date' => request('date-range'),
            'from-date' => request('from-date'),
            'to-date' => request('to-date'),
            'make_model' => request('make_and_model'),
            'manufacturer' => request('manufacturer'),
        ];
    }

    /**
     * Export the data to CSV.
     *
     * @return array The exported data in CSV format.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();
        $assets =  $this->service->exportData();

        $assets->chunk(5000, function ($assetChunks) use ($data, $datas) {
            $datas->push($this->service->getReportExportData($assetChunks, $data));
        });

        return $datas->toArray();
    }

    /**
     * Export the graph with list data.
     *
     * @throws Exception Some error ocurred. Please try again later
     * @return mixed
     */
    public function exportGraphWithListData()
    {
        try {
            setUnlimitedExecutionTimeAndMemoryLimit();

            $dataCollection = $this->export();
            $excel = $this->service->exportGraphAndDataListToExcel($dataCollection);

            return $excel;
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }
    }
}
