<?php

namespace App\Http\Controllers\Reports\CycleCount;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\AssetType;
use App\Models\Location;
use App\Models\ReportHistoryVarience;
use App\Models\CycleCountScannedData;
use Facades\App\Services\Reports\CycleCount\CycleCountFileGeneratorService;
use Facades\App\Services\Reports\CycleCount\CycleCountReportService;
use Facades\App\Services\Reports\CycleCount\CycleCountScanService;
use Facades\App\Services\Reports\CycleCount\CycleCountHistoryService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class CycleCountController extends Controller
{
    /**
     * @var array The asset types to be excluded from operations.
     */
    public $excludedAssetTypes;

    /**
     * @var array The asset statuses to be excluded from operations.
     */
    public $excludedAssetStatuses;

    /**
     * Initialize the controller with excluded asset types and statuses.
     *
     * @return void
     */
    public function __construct()
    {
        $this->excludedAssetTypes = config('cycle-count.excluded-asset-types');
        $this->excludedAssetStatuses = config('cycle-count.excluded_asset_statuses');
    }

    /**
     * Display the cycle count report view with necessary data.
     *
     * @return \Illuminate\View\View The view for the cycle count report with necessary data
     */
    public function index()
    {
        $locations = Location::freezed()->orderBy('updated_at', 'asc')->paginate(50);
        $locationId = null;

        $assetTypes = AssetType::orderBy('slug')
            ->whereNotIn('slug', $this->excludedAssetTypes)
            ->get();

        return view('reports.cycle-count.cycle-count', compact('locations', 'locationId', 'assetTypes'));
    }

    /**
     * Update rows in the cycle count scan process.
     *
     * @return \Illuminate\Http\JsonResponse|string|bool JSON response with the rendered view,
     *         a string indicating the error, or false if required parameters are missing
     */
    public function updateRows()
    {
        $assetTypeId = CycleCountScanService::getSelectedAssetTypes();
        $serialNo = request('serial_no');

        if (!$assetTypeId && !$serialNo) {
            return false;
        }

        $reqAsset = CycleCountScanService::getReqAssetDetails($serialNo, $assetTypeId);

        if ($reqAsset == 'type_mismatch') {
            return $reqAsset;
        }

        $reqAsset = CycleCountScanService::checkExistInScannedList($serialNo, $assetTypeId);

        if ($reqAsset == 'serial_exist') {
            return $reqAsset;
        }

        $scannedAsset = CycleCountScanService::createVarianceScannedData($assetTypeId, $serialNo);

        if (!$scannedAsset) {
            return 'asset_error';
        }

        $view = view('reports.cycle-count.partials.added-variance', ['scannedData' => '', 'scannedAset' => $scannedAsset])->render();

        return response()->json($view);
    }

    /**
     * Merge data for the cycle count report based on the location and asset type.
     *
     * @param int|null $locationId  The ID of the location (optional, defaults to null)
     * @param int|null $assetTypeId The ID of the asset type (optional, defaults to null)
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\View\View Redirects back with an error if location ID is empty, otherwise returns the view with merged data
     */
    public function mergeData($locationId = null, $assetTypeId = null)
    {
        if (empty($locationId)) {
            return redirect()->route('cycle-count.index')->with('error', 'Location is empty');
        }

        $assetTypes = AssetType::orderBy('slug')
            ->whereNotIn('slug', $this->excludedAssetTypes)
            ->get();

        $locations = Location::freezed()
            ->orderBy('updated_at', 'asc')
            ->paginate(50);

        $mergedData = CycleCountScannedData::whereDate('created_at', '=', Carbon::today()->toDateString())
            ->where('location_id', $locationId)
            ->get();

        // Get all asset type ids used in the scanned data.
        $assetTypeIds = CycleCountScanService::getScannedAssetTypesIds($mergedData);
        $assetTypeId = implode(',', $assetTypeIds);
        $selectedLocationId = $locationId;

        return view('reports.cycle-count.cycle-count', compact('mergedData', 'assetTypeIds', 'assetTypeId', 'locations', 'locationId', 'assetTypes', 'selectedLocationId'));
    }

    /**
     * Generate the cycle count report.
     *
     * @param integer request('count')           Total number of scanned Asset
     * @param integer request('location_id')     Scanned location ID
     * @param string request('serial_no')        Scanned Serial #/ Asset Tag #. From this string we will make an array of assetTags.
     *
     * @return \Illuminate\View\View The view with the cycle count report results
     */
    public function generate()
    {
        $count = request('count');
        $assetTypeIdString = request('asset_type_id') ? request('asset_type_id') : '';
        $assetTypeId = request('asset_type_id') ? explode(",", $assetTypeIdString) : [];

        $allSerialNos = [];
        $location = Location::find(request('location_id'));

        for ($i = 1; $i <= $count; $i++) {
            $allSerialNos[] = request('serial_no' . $i);
        }

        $serialNos = CycleCountReportService::excludeAssetsFromAssetTagsArray($allSerialNos, $assetTypeId);

        // Set selected asset types as empty if "All" is selected.
        if (in_array("All", $assetTypeId)) {
            $assetTypeId = [];
        }

        $reportName = $location->room_name . " " . date('Y-m-d H:i:s');

        $datas = CycleCountReportService::getScannedAssetData($serialNos, $location);
        $datasNotScanned = CycleCountReportService::getNotScannedAssetData($serialNos, $location, $assetTypeId);
        $dataNotInDatabase = CycleCountReportService::getNotInDatabaseAssetData($serialNos, $location);
        $assetsNotInDatabase = count($dataNotInDatabase);

        // Comented Code: dd($datas, $datasNotScanned, $dataNotInDatabase);.
        $assetsMatched = CycleCountReportService::getAssetsMatchedCount($serialNos, $location->id, $assetTypeId);
        $assetsNotMatched = CycleCountReportService::getAssetsNotMatchedCount($serialNos, $location->id, $assetTypeId);
        $assetsScannedButInAnotherLocation = CycleCountReportService::getAssetsScannedButInAnotherLocationCount($serialNos, $location->id, $assetTypeId);

        // Sum of assets scanned but not in db or assets scanned but in another location or in assigned status is the positive variance.
        $positiveVariance = $assetsScannedButInAnotherLocation + $assetsNotInDatabase;

        // Commented Unused: $assetsNotScanned = $datasNotScanned->count();.
        $assetsNotMatched = ($count - $assetsMatched) + $assetsNotMatched;

        $count = $assetsMatched + $assetsNotMatched;
        $accuracy = $assetsMatched > 0 ? round(($assetsMatched / ($count) * 100), 2) : 0;
        $variance = 100 - $accuracy;

        // Get comma seperated name of all scanned users.
        $userName = CycleCountReportService::getUserNamesForVarianceReport(request('location_id'), Carbon::today()->toDateString()) ?: Auth::user()->userName ?? '';

        $userId = Auth::id();
        $createdDate = parse_date_from_db_datetime(Carbon::now());

        // Commented Unused: $roomName = $location->room_name;.
        $totalScanned = request('count') ?: 0;

        // Generate csv file.
        $reportCsvFile = CycleCountFileGeneratorService::generateCsvFile($datas, $datasNotScanned, $dataNotInDatabase, $assetsMatched, $assetsNotMatched, $accuracy, $reportName, $createdDate, $userName, $positiveVariance, $variance, $count);

        // Create variance report.
        $reportId = CycleCountHistoryService::saveVarienceReport($datasNotScanned, $assetsMatched, $assetsNotMatched, $assetsNotInDatabase, $count, $accuracy, $userName, $createdDate, $userId, $reportName, $totalScanned, $location->id, $location->room_name, $reportCsvFile, $assetTypeIdString, $variance, $positiveVariance);

        // Commented: $reportCsvFile = '';.
        // Clear the `cycle_count_scanned_datas` table after generating the report.
        CycleCountScannedData::whereDate('created_at', '=', Carbon::today()->toDateString())->where('location_id', request('location_id'))->delete();

        // Unfreeze location after generating the report.
        $location->update(['is_freeze' => 0]);

        // Remove token to prevent refreshing the page.
        request()->session()->remove('_token');

        return view('reports.cycle-count.cycle-count-result', compact('datas', 'datasNotScanned', 'dataNotInDatabase', 'assetsMatched', 'assetsNotMatched', 'count', 'variance', 'positiveVariance', 'accuracy', 'userName', 'createdDate', 'reportName', 'location', 'reportCsvFile', 'reportId'));
    }

    /**
     * Download the variance report CSV file.
     *
     * @param int $id The ID of the variance report
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response to download the CSV file
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If the report ID is not found
     */
    public function downloadReport($id)
    {
        $varianceReport = ReportHistoryVarience::find($id);

        return response()->download(storage_path('app/public/' . $varianceReport->report_csv));
    }

    /**
     * Get the freeze status of a location.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the freeze status of the location
     * @throws \Illuminate\Validation\ValidationException If the location ID is not provided
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If the location ID is not found
     */
    public function getLocationFreezeStatus()
    {
        request()->validate([
            'locationId' => 'required'
        ]);

        $locationId = request('locationId');
        $location = Location::find($locationId);

        return response()->json($location->is_freeze);
    }

    /**
     * Change the freeze status of a location.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the updated view and freeze status
     * @throws \Illuminate\Validation\ValidationException If the location ID is not provided
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If the location ID is not found
     */
    public function changeLocationFreezeStatus()
    {
        request()->validate([
            'locationId' => 'required'
        ]);

        $locationId = request('locationId');
        $location = Location::find($locationId);

        if (request('id') == 0) {
            $location->update(['is_freeze' => 1]);
        }

        if (request('id') == 1) {
            $location->update(['is_freeze' => 0]);
        }

        Cache::forget('warehouse-locations');
        $locations = Location::freezed()->orderBy('updated_at', 'asc')->paginate(50);

        $data['view'] = view('reports.cycle-count.partials.frozen-locations', ['locations' => $locations])->render();
        $data['is_freeze'] = $location->is_freeze;

        return response()->json($data);
    }

    /**
     * Get the scanned assets for the current user and given location.
     *
     * @param integer request('location_id') User selected locaion for scan
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered view and count of scanned assets
     */
    public function scannedAssets()
    {
        $scannedData = CycleCountScannedData::whereDate('created_at', '=', Carbon::today()->toDateString())
            ->where('location_id', request('location_id'))
            ->where('user_id', Auth::id())->get();

        $data['view'] = view('reports.cycle-count.partials.added-variance', ['scannedData' => $scannedData])->render();
        $data['count'] = $scannedData->count();

        return response()->json($data);
    }

    /**
     * Delete scanned assets based on the request parameters.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return bool|null True if deletion was successful, null if the asset was not found
     */
    public function deleteScannedAssets(Request $request)
    {
        if (!$request->id) {
            return CycleCountScannedData::where('location_id', $request->locationId)->where('serial_no', $request->assetTagId)->delete();
        }

        return CycleCountScannedData::find($request->id)->delete();
    }

    /**
     * Check if a serial number exists in the scanned assets list.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating whether the serial number exists in the scanned list
     */
    public function checkScannedAssets()
    {
        $serialNo = request('serial_no');
        $reqAsset = CycleCountScanService::checkExistInScannedList($serialNo);

        return response()->json($reqAsset);
    }

    /**
     * Check the asset type of a scanned serial number.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the asset details
     */
    public function checkScannedAssetType()
    {
        $assetTypeId = CycleCountScanService::getSelectedAssetTypes();
        $serialNo = request('serial_no');
        $reqAsset = CycleCountScanService::getReqAssetDetails($serialNo, $assetTypeId);

        return response()->json($reqAsset);
    }
}
