<?php

namespace App\Http\Controllers;

use App\Events\BulkUpdates;
use App\Http\Requests\CsvFileUploadRequest;
use Illuminate\Http\Request;
use Facades\App\Services\AssetBulkUpload;
use App\Models\AssetStatus;
use Facades\App\Repositories\AssetRepository;
use App\Models\Asset as AssetModel;
use Facades\App\Models\MakeAndModel;
use Illuminate\Support\Facades\Auth;
use App\User;
use App\Models\AssetType;
use App\Models\Location;
use Facades\App\Services\JiraService;
use Facades\App\Services\AssetData;
use Facades\App\Services\Asset\AssetStatusService;
use App\Http\Requests\StoreAssetRequest;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\UserType;
use Cache;
use Carbon\Carbon;

class NetworkAssetController extends Controller
{


    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $statuses = AssetStatus::orderBy('slug')->get();

        return view('assets.asset-network', compact('statuses'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $assetTypes = AssetType::network();
        $data = ['title' => 'Create Network Asset', 'parent' => 'Asset', 'type' => 'network', 'parentUrl' => '/search-network-assets'];

        return view('assets.create-network-asset', compact('assetTypes', 'data'));
    }

    public function add(StoreAssetRequest $request, $count = 1)
    {
        if (session('count_add')) {
            $count = session('count_add') + 1;
        }

        session(['count_add' => $count]);
        $asset_status = AssetStatusService::whenCreatingAsset();
        $request['asset_status_id'] = AssetStatus::where('slug', $asset_status)->first()->id;
        $request['count_add'] = session('count_add');
        $request['asset_tag'] = AssetBulkUpload::getAssetTag($request['asset_tag']);

        $view = view('assets.partials.added-network-assets', compact('request'))->render();

        return response()->json($view);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return \Illuminate\Http\Response
     */
    public function store()
    {
        $count = request('count');

        $assetStatus = AssetStatusService::whenCreatingAsset();

        for ($i = 1; $i <= $count; $i++) {

            if (!request()->has('ticket_no' . $i)) continue; //row was deleted before submitting

            $data = $this->getData($i);
            $assetStatusId = AssetStatus::where('slug', $assetStatus)->first()->id;

            $location = Location::where('id', $data['location_id'])->with('locationType')->first();

            if ($location->locationType->location_type_name == 'Install') {
                $assetStatusId = AssetStatus::where('slug', 'installed')->first()->id;
            }

            $data['asset_status_id'] = $assetStatusId;

            $asset_id = AssetModel::updateorCreate(['asset_tag' => $data['asset_tag']], $data)
                ->id;

            $user = User::find(Auth::id());
            $description = __('history.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'newroomname' => $location->room_name,
            ]);

            if ($location->locationType->location_type_name == 'Install') {
                $description = __('history.CreatedAndInstalled', [
                    'assetname' => $data['serial_no'],
                    'assetid' => $asset_id,
                    'room_name' => $location->room_name,
                ]);
            }

            $jiraDescription = __('jira.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'username' => $user->first_name . ' ' . $user->last_name,
                'useremail' => $user->email,
            ]);

            $jiraData[] = [
                'description' => $jiraDescription,
                'ticketId'   => request('ticket_no' . $i),
                'userId' => $user->id,
            ];

            $assetHistory = [
                'user_id' => Auth::id(),
                'asset_id' => $asset_id,
                'ticket_no' => request('ticket_no' . $i),
                'ticket_service_provider' => config('ticket-integration.service'),
                'action' => 'created',
                'new_location_id' => $data['location_id'],
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($assetHistory));
        }
        session(['count_add' => 0]);

        JiraService::bulkSendToJira($jiraData);

        return redirect(route('network-assets.index'))->with('message', 'Assets created successfully.');
    }

    public function show($id)
    {
        $asset = AssetModel::with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'parentAsset', 'carrier', 'assetHistory:id,action,user_id,asset_id,created_at,updated_at,ticket_no')->find($id);

        $hardwares = MakeAndModel::getAll();
        $data = ['title' => 'Network Asset Detail', 'parent' => 'Network Assets', 'type' => 'network', 'parentUrl' => '/network-assets'];

        return view('assets.detail', compact('asset', 'hardwares', 'data'));
    }


    public function getData($count)
    {
        $assetData['po_id'] = request('po_id' . $count);
        $assetData['asset_tag'] = request('asset_tag' . $count);
        $assetData['location_id'] = request('location_id' . $count);
        $assetData['asset_type_id'] = request('asset_type_id' . $count);
        $assetData['make_and_model_id'] = request('make_and_model_id' . $count);
        $assetData['technical_spec_id'] = request('technical_spec_id' . $count);
        $assetData['serial_no'] = request('serial_no' . $count);
        $assetData['ticket_no'] = request('ticket_no' . $count);
        $assetData['ip'] = request('ip' . $count);
        $assetData['mac'] = request('mac' . $count);

        $serialNo = $assetData['serial_no'];
        if (($serialNo[0] == 'S') || ($serialNo[0] == 's')) {
            $assetData['serial_no'] = substr($serialNo, 1);
        }

        return $assetData;
    }

    /**
     * Upload a CSV file containing network asset data and store it in the `asset_bulk_upload` directory.
     * This function validates the file extension, imports the CSV data, validates the columns, validates the fields, and checks for duplicate values.
     * If any errors are found, it renders the errors as HTML and returns the response as JSON.
     * If the data is valid, it renders the data as HTML and returns the response as JSON.
     *
     * @param CsvFileUploadRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function upload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $path = 'network-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_upload', $path);

        $path = storage_path('app/' . $path);

        $view = [];

        $count = session('count_add');
        $data = AssetBulkUpload::importNetworkAssetData($path, $count);

        if (!empty($data['error'])) {
            $errors = $data['error'];
            $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();
            return response()->json($view);
        }

        $csvData = $data['csvData'];


        $errors = AssetBulkUpload::validateCsvColumns($path, $csvData, 'network');
        $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();

        if ($errors != '') {
            return response()->json($view);
        }

        if ($csvData) {
            $errors = AssetBulkUpload::validateNetworkFields($csvData);
            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            if (!empty(array_filter($errors))) {
                return response()->json($view);
            }

            $duplicateErrors = AssetData::checkRepeatNetworkValues($csvData);
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            if (!empty($duplicateErrors)) {
                $csvData = [];
            }
            $view['data'] = view('assets.partials.upload-network-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    public function data(Request $request)
    {
        $filteredData   = AssetData::filter();
        $assets         = $filteredData['assets'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($assets)) {
            $data = AssetData::getNetworkAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    public function search()
    {
        $breadcrumbTitle = 'Manage Assets';
        $user = '';
        $location = '';
        $type = request('general_type');
        $searchedLocationAssetTagCount = 0;
        $searchedUserAssetTagCount = 0;

        $installLocations = Cache::remember('install-locations', 720, function () {
            return Location::withoutwarehouse()->active()->orderBy('room_name', 'asc')->get();
        });

        $warehouseLocations = Cache::remember('warehouse-locations', 720, function () {
            return Location::warehouses()->active()->orderBy('room_name', 'asc')->get();
        });

        $installAndBuildingLocations = Cache::remember('install-building-locations', 720, function () {
            return Location::installBuildings()->active()->orderBy('room_name', 'asc')->get();
        });

        if (request()->has('searchText') && !empty(request('searchText'))) {
            $user = AssetRepository::searchAssetUser(request('searchText'), 'network');
            $location = AssetRepository::searchAssetLocation(request('searchText'), 'network');

            $query = AssetModel::networkAsset();
            $searchedLocationAsset = AssetRepository::assetLocationSearched($query, request('searchText'))->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->first();


            $query = AssetModel::networkAsset();
            $searchedUserAsset = AssetRepository::assetUserSearched($query, request('searchText'))->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->first();

            if ($searchedLocationAsset) {
                $searchedLocationAssetTagCount = 1;
            }

            if ($searchedUserAsset) {
                $searchedUserAssetTagCount = 1;
            }

            if ($location) {
                $location->assets = $location->assets()->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->networkAsset()->notSearched(request('searchText'));
                $location->assets = $location->assets->paginate(50);

                if ($location->assets->currentPage() == 1 && $searchedLocationAsset) {
                    $location->assets->prepend($searchedLocationAsset);
                }
            }

            if ($user) {
                $user->assets = $user->assets()->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->networkAsset()->notSearched(request('searchText'));
                $user->assets = $user->assets->paginate(50);

                if ($user->assets->currentPage() == 1 && $searchedUserAsset) {
                    $user->assets->prepend($searchedUserAsset);
                }
            }
        }
        $statuses = AssetStatus::where('slug', '!=', 'assigned')->orderBy('slug')->get();
        request()->flash();

        $userTypes = ['Super User', 'Super Admin'];
        $userTypeName = UserType::find(User::find(Auth::id())->user_type_id)->name;
        $adminStatus = 0;

        if (in_array($userTypeName, $userTypes)) {
            $adminStatus = 1;
        }

        return view('assets.search-asset', compact('user', 'location', 'statuses', 'breadcrumbTitle', 'searchedUserAssetTagCount', 'searchedLocationAssetTagCount',  'type', 'installLocations', 'warehouseLocations', 'installAndBuildingLocations', 'adminStatus'));
    }
}
