<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\User;
use Illuminate\Support\Facades\Notification;
use App\Notifications\AuthenticateUser;
use App\Notifications\Slack;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use App\Http\Traits\ThrottlesSuccessfullLogins;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Exception;

class LoginController extends Controller
{

    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use ThrottlesLogins;
    use ThrottlesSuccessfullLogins;

    protected $maxAttempts = 3; // Number of login attempts allowed.
    protected $decayMinutes = 2; // Time in minutes until the login attempts are reset.
    protected $maxSuccessfulAttempts = 5; // Number of login attempts allowed.
    protected $successDecayMinutes = 2; // Time in minutes until the login attempts are reset.

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Authenticates a user based on the given request.
     *
     * @param Request $request The HTTP request object.
     * @throws Some_Exception_Class A description of the exception that can be thrown.
     * @return Some_Return_Value The return value of the function.
     */
    public function authenticate(Request $request)
    {
        $this->validateLogin($request);

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        if (method_exists($this, 'hasTooManyLoginAttempts') && $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);

            // Log the lockout event.
            Log::channel('daily')->info('Login Lockout from IP ' . \Request::ip() . ' for Email ' . $request->email);

            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request)) {
            return view('auth.login-verify', ['email' => $request->email]);
        }

        // Failed login attempt.
        $this->incrementLoginAttempts($request);

        Log::channel('daily')->info('Login Failed from ip ' . \Request::ip() . ' with Email ' . $request->email);

        return redirect(route('login'))->withInput($request->only('email', 'remember'))->withErrors(['password' => 'Invalid Credentials']);
    }

    /**
     * Validate the user login request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function validateLogin(Request $request)
    {
        $this->validate($request, [
            'email' => 'required|email',
            'password' => 'required',
        ]);
    }

    /**
     * Attempt to log the user into the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return bool
     */
    protected function attemptLogin(Request $request)
    {
        $email    = $request->email;
        $password = $request->password;
        array_get('password', $request);
        $user     = User::where('email', $email)->first();

        if (!empty($user) && Hash::check($password, $user->password)) {
            // To rate limit the maximum succesful login attempts.
            if (method_exists($this, 'hasTooManySuccessfulLoginAttempts') && $this->hasTooManySuccessfulLoginAttempts($request)) {
                $this->fireSuccessLockoutEvent($request);

                // Log the lockout event.
                Log::channel('daily')->info('Success Login Lockout from IP ' . \Request::ip() . ' for Email ' . $request->email);

                return $this->sendSuccessLockoutResponse($request);
            }

            $this->incrementSuccessLoginAttempts($request);

            $user->update(['email_token' => bin2hex(openssl_random_pseudo_bytes(4))]);
            Notification::send($user, new AuthenticateUser($user));

            return true;
        }

        return false;
    }

    // Specify the username field here  we have email so for the throttle functions we need to specify the username field
    public function username()
    {
        return 'email';
    }

    /**
     * Verify user email.
     *
     * @param Request $request The HTTP request object.
     * @return \Illuminate\Http\RedirectResponse
     */
    public function emailVerify(Request $request)
    {
        $token = $request->email_token;

        if (!$token) {
            return  redirect()->route('login')->withErrors(['email' => 'Email Verification Token not provided!']);
        }

        $user = User::whereEmail($request->email)->where('email_token', $token)->first();

        if (!$user) {
            return  redirect()->route('login')->withErrors(['flash-error' => 'Invalid Email Verification Token!']);
        }

        Auth::login($user);
        Log::channel('daily')->info('User logged in from ip ' . \Request::ip() . ' with Email ' . $request->email);

        $request->session()->regenerate();

        // If we need to clear lockout after succesfull do enable this.
        // $this->clearLoginAttempts($request);

        if ($user->hasPermissionTo('IT Assets View')) {
            return redirect(route('assets.index'));
        }

        return redirect(route('my-teq-assets'));
    }

    /**
     * Logs out the current user.
     *
     * @throws Some_Exception_Class description of exception
     * @return void
     */
    public function logout()
    {
        Auth::logout();

        return redirect('login');
    }

    /**
     * Send the logged in user details to Slack notification if the user is Admin or Super admin
     * 
     * @param $user App\User
     */
    public function sendLoggedInNotification($user)
    {
        $userType = optional($user->userType)->name;

        // if (!in_array($userType, ['Administrator', 'Super Admin'])) {
        //     return true; // no need to send notification
        // }

        $message = "$user->email ($userType) Logged in to " . url('/');

        try {
            Notification::route('slack', config('teqtivity-notification-mails.login_slack'))->notify(new Slack($message));
        } catch (Exception $e) {
            Log::channel('daily')->info("Slack Notification error: " . $e->getMessage());
        }
    }
}
