<?php

namespace App\Http\Controllers\Location;

use App\Http\Controllers\Controller;
use Facades\App\Services\SearchAsset;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class LocationFilterFieldsController extends Controller
{

    /**
     * Return the locations that match the given search criteria.
     *
     * @param string $search The search criteria.
     * @param string $type The type of location to search for (room_name, building, city).
     *
     * @return \Illuminate\Http\JsonResponse The locations that match the search criteria.
     */
    public function getFilterLocationFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {
            $result = SearchAsset::getFilterLocationFields($search, $type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }

    /** Get the locations based on relation
     * @param Request $request
     *
     * @return [type]
     */
    public function getFilterLocationFieldsWithRelation(Request $request)
    {
        $request->validate([
            'search' => ['required'],
            'field'  => Rule::in(['room_name', 'building', 'city'])
        ]);

        $result = SearchAsset::getFilterLocationFields(request('search'), request('field'), request('relation'));

        if (!$result->count()) {
            $result[] = ['id' => '0', 'text' => 'No result found'];
        }

        return response()->json($result);
    }

    /** Get the storage locations based on search
     * @return [type]
     */
    public function getFilterStorageLocation()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = SearchAsset::getFilterStorageLocation($search);

            if (empty($result)) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }

    /** Get the install locations based on search
     * @return [type]
     */
    public function getFilterInstallLocation()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = SearchAsset::getFilterInstallLocation($search);

            if (empty($result)) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }

    /** Get the install & building locations based on search
     * @return [type]
     */
    public function getFilterInstallAndBuildingLocations()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = SearchAsset::getFilterInstallAndBuildingLocations($search);

            if (empty($result)) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }

    /** Get the installed locations based on search
     * @return [type]
     */
    public function getFilterInstalledLocations()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = SearchAsset::getFilterInstalledLocations($search);

            if (empty($result)) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }
}
