<?php

namespace App\Http\Controllers\Location;

use App\Http\Controllers\Controller;
use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Models\Location;
use App\Models\Asset;
use App\Http\Requests\StoreLocation;
use App\User;
use Illuminate\Support\Facades\Auth;
use Facades\App\Services\AssetHistory as RepoAssetHistory;
use App\Services\Location\LocationData;
use Carbon\Carbon;
use App\Http\Responses\ReportJsonResponse;
use Facades\App\Services\ExportManagement\ExportManagementService;
use Illuminate\Support\Facades\Cache;

class LocationController extends Controller
{

    /**
     * Constructor for the SomeService class.
     *
     * @param LocationData $service Service for managing location data.
     */
    public function __construct(protected LocationData $service) {}

    /**
     * Display the location index page.
     *
     * @return \Illuminate\View\View The view for the location index page.
     */
    public function index()
    {
        return view('location.index');
    }

    /**
     * Display the details of a specific location.
     *
     * @param int $locationId The ID of the location to display.
     *
     * @return \Illuminate\View\View The view for the location detail page.
     */
    public function show($locationId)
    {
        $location = Location::withCount(['assets', 'accessories', 'zoomDevices'])->with(['locationType', 'country', 'region', 'state', 'siteCode'])->find($locationId);

        return view('location.detail', compact('location'));
    }

    /**
     * Display the form for creating a new location.
     *
     * @return \Illuminate\View\View The view for the location creation form.
     */
    public function create()
    {
        return view('location.create');
    }

    /**
     * Display the form for editing a specific location.
     *
     * @param int $locationId The ID of the location to edit.
     *
     * @return \Illuminate\View\View The view for the location edit form.
     */
    public function edit($locationId)
    {
        $location = Location::withCount('assets')->with(['locationType', 'siteCode.region', 'siteCode.country'])->findOrFail($locationId);
        $assets = Asset::with('assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus')->where('location_id', $locationId)->paginate(25);

        return view('location.edit', compact('location', 'assets'));
    }

    /**
     * Store a newly created location in the database.
     *
     * @param StoreLocation $request The validated request instance.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the location index page with a success message.
     */
    public function store(StoreLocation $request)
    {
        $request = $request->only(['location_type_id', 'room_name', 'site_code_id', 'building', 'address', 'floor', 'country_id', 'region_id', 'status']);

        $request['manually_created'] = 1;

        Location::create($request);

        $description = __('history.LocationCreated', [
            'roomName' => request('room_name')
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'location_added',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));

        Cache::forget('install-locations');
        Cache::forget('warehouse-locations');

        return redirect(route('location.index'))->with('message', 'Location created successfully.');
    }

    /**
     * Update the specified location in the database.
     *
     * @param StoreLocation $request  The validated request instance.
     * @param int           $location The ID of the location to update.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the location index page with a success message or an error message.
     */
    public function update(StoreLocation $request, $location)
    {
        $locationSet = Location::with(['locationType', 'region', 'country'])
            ->where('id', $location)
            ->first();

        if (request('status') == 0 && Location::find($location)->assets()->count()) {
            return redirect(route('location.edit', ['location' => $location]))->with('error', 'Remove all assets from this location before deactivating');
        }

        $request = $request->only(['location_type_id', 'room_name', 'site_code_id', 'building', 'address', 'floor', 'country_id', 'region_id', 'status']);

        Location::findOrFail($location)->update($request);

        Cache::forget('install-locations');
        Cache::forget('warehouse-locations');

        $newLocationSet = Location::with(['locationType', 'region', 'country'])
            ->where('id', $location)
            ->first();

        $description = '';

        // To save location history.
        $description = RepoAssetHistory::getLocationDescription($locationSet, $request, $newLocationSet, $comments = null);

        if ($description != '') {
            $description = 'Update Location ' . $locationSet->room_name . ':' . $description;

            $assetHistory = [
                'user_id' => Auth::id(),
                'new_location_id' => $location,
                'action' => 'location_updated',
                'description' => $description,
                'new_value' => 'Location',
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($assetHistory));
        }

        return redirect(route('location.index'))->with('message', 'Location updated successfully.');
    }

    /**
     * Retrieve and return filtered location data for the report.
     *
     * @param Request $request The incoming request instance.
     *
     * @return ReportJsonResponse JSON response with the filtered data for the report.
     */
    public function data(Request $request)
    {
        $filteredData = $this->service->getData();
        $locations = $filteredData['locations'];
        $start = request('start');
        $data = [];

        if (!empty($locations)) {
            $data = $this->service->getLocationData($locations, $start, $data);
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export location data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse|string The response containing the CSV file or a message indicating background report generation.
     */
    public function export()
    {
        $locationAll = $this->service->getExportData();

        if (ExportManagementService::locationLargeExport($locationAll)) {
            return 'Your report will be generated in the background. Go to the <a target="_blank" href="/generated-exports/location_report">Reports Page</a> to download the report.';
        }

        setUnlimitedExecutionTimeAndMemoryLimit();

        $datas = collect();
        $data = [];
        $datas->push($this->service->getLocationExportData($locationAll->get(), 0, $data));

        return exportToCsv($datas->toArray());
    }

    /**
     * Retrieves the information of a single location by its ID.
     *
     * @param int $id The ID of the location.
     * @return \Illuminate\Http\JsonResponse The JSON response containing the location information.
     */
    public function getSingleLocationInfo($id)
    {
        if (!$id) {
            return response()->json([]);
        }
        $locationInfo = Location::with(['country:id,name', 'state:id,name'])->where('id', $id)->first();
        return response()->json($locationInfo);
    }
}
