<?php

namespace App\Http\Controllers\InventoryConsumption;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Services\InventoryConsumption\ConsumptionReportService;
use Illuminate\Http\Request;

class ConsumptionReportController extends Controller
{
    
    protected $service;

    /**
     * Constructor for ConsumptionReportController.
     *
     * @param ConsumptionReportService $service The service class instance.
     */
    public function __construct(ConsumptionReportService $service)
    {
        $this->service = $service;
    }

    /**
     * Show the Inventory Consumption Report view.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $filterDataOptions = $this->service->getFilterDataOptions();
        
        return view('reports.inventory-consumption.consumption-report', compact('filterDataOptions'));
    }

    /**
     * Fetch data for the Inventory Consumption Report.
     *
     * @param Request $request The HTTP request object.
     * @return \Illuminate\Http\JsonResponse
     */
    public function data(Request $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $filteredData = $this->service->data();
        $assets = $filteredData['assets'];
        $data = [];

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export the Inventory Consumption Report data to a downloadable format.
     *
     * @return \Illuminate\Http\Response
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $datas = collect();
        $assets = $this->service->exportData();
        $data = [];

        $assets->chunk(5000, function ($assetChunks) use ($data, $datas){
            $datas->push($this->service->getReportExportData($assetChunks, $data));
        });

        return exportToCsv($datas->toArray());
    }
}
