<?php

namespace App\Http\Controllers;

use App\Models\AssetType;
use App\Models\AssetTypeAttribute;
use App\Models\Carrier;
use App\Models\Country;
use App\Models\TechnicalSpecs;
use App\Services\FilterFieldService;
use Exception;
use Facades\App\Repositories\HardwareStandards;
use Facades\App\Services\SearchAsset;
use Illuminate\Support\Facades\Log;

class FilterFieldController extends Controller
{
    /**
     * A constructor for the class.
     *
     * @param FilterFieldService $service The service parameter for the constructor
     */
    public function __construct(protected FilterFieldService $service) {}

    /**
     * Fetch teqtivity users based on search.
     * @return json
     */
    public function getFilterTeqtivityUser()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = $this->service->getFilterTeqtivityUser($search);

            return response()->json($result);
        }
    }

    /**
     * Get technical specs for dropdown fields
     *
     * @throws Exception description of exception
     * @return JSON
     */
    public function getTechnicalSpecs()
    {
        try {
            $result = $this->service->getTechnicalSpecs();

            return response()->json($result);
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }
    }
    /**
     * Get Hardware Standards for dropdown fields
     *
     * @throws Exception
     * @return JSON
     */
    public function getHardwareStandards()
    {
        try {
            $result = $this->service->getHardwareStandards();

            return response()->json($result);
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }
    }

    /**
     * Retrieves all users with Teqtivity users fields based on a search query.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the search results.
     */
    public function getUsersWithTeqtivityUsersFields()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = $this->service->getFilterUserWithTeqtivityUsersFields($search);

            return response()->json($result);
        }
    }

    /**
     * Lists all hardware standards with IDs and names that can be used to populate a dropdown
     * for creating an asset.
     *
     * @return \Illuminate\Support\Collection A collection of arrays containing the hardware standard IDs and names.
     */
    public function listHardwareCreate()
    {
        $makeAndModel = HardwareStandards::getHardwareStandardsCreate(request('id'), request('type'))->get();

        return $makeAndModel->map(
            function ($item) {
                return [
                    'id' => $item->id,
                    'hardware' => $item->makemodelName
                ];
            }
        );
    }

    /**
     * Lists technical specifications for a specific make and model to populate a dropdown for creating an asset.
     *
     * @return \Illuminate\Database\Eloquent\Collection A collection of technical specifications ordered by details.
     */

    public function listTechnicalSpecCreate()
    {
        return TechnicalSpecs::with(['makeAndModel'])->active()->where('make_and_model_id', request('id'))->orderBy('details', 'ASC')->get();
    }

    /**
     * Lists carriers for a dropdown based on asset type.
     *
     * If the asset type is 'mobile_device', all carriers are listed.
     * Otherwise, carriers with the slug 'unlocked' are excluded.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the carriers as HTML option elements.
     */

    public function listCarrier()
    {
        $carriers = Carrier::all();

        return $carriers->map(
            function ($item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name
                ];
            }
        );
    }

    /**
     * Lists additional attributes for an asset type, which can be used to search assets.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the additional attributes as HTML form elements.
     */
    public function listAdditionalSearchAttribute()
    {
        $attributes = AssetTypeAttribute::GetAttributesByAssetType(request('id'))->get();

        $attributesView = view('assets.partials.search-extra-attributes', compact('attributes'))->render();

        return response()->json($attributesView);
    }

    /**
     * Lists filter fields for a dropdown based on search and type.
     *
     * Used in the asset search form to populate the filter fields.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filter fields as HTML option elements.
     */
    public function getAssetFilterFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getAssetFilterFields($search, $type);

            if (!count($result)) {
                $result[] = ['value' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Fetch data to populate the fields with assignable assets.
     * @return json
     */
    public function getFilterAssetsForAssignFields()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = SearchAsset::getSearchAssetForAssignFields($search);

            return response()->json($result);
        }
    }

    /**
     * Lists filter fields for a dropdown based on search and type for server assets.
     *
     * Used in the asset search form to populate the filter fields for server assets.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filter fields as HTML option elements.
     */
    public function getFilterServerFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getFilterTicketServerFields($search, $type);

            if (!$result->count()) {
                $result[] = ['value' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Lists filter fields for a dropdown based on search and type for network assets.
     *
     * Used in the asset search form to populate the filter fields for network assets.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filter fields as HTML option elements.
     */
    public function getFilterNetworkFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getFilterTicketNetworkFields($search, $type);

            if (!$result->count()) {
                $result[] = ['value' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Lists filter fields for a dropdown based on search and type for Workspace ONE assets.
     *
     * Used in the asset search form to populate the filter fields for Workspace ONE assets.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filter fields as HTML option elements.
     */
    public function getFilterFieldsWorkspaceOne()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getFilterWorkspaceOneFields($search, $type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Lists filter fields for a dropdown based on search and type for Chromebook assets.
     *
     * Used in the asset search form to populate the filter fields for Chromebook assets.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filter fields as HTML option elements.
     */
    public function getFilterFieldsChromebook()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getFilterChromebookFields($search, $type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Populate filter dropdowns for Intune reports
     * @return [type]
     */
    public function getFilterFieldsIntune()
    {

        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = SearchAsset::getFilterIntuneFields($search, $type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Autocomplete for countries.
     * @return \Illuminate\Http\JsonResponse List of countries matching the search criteria.
     */
    public function getCountryList()
    {
        $search = request('search');

        if (!empty($search)) {
            $result = Country::where('name', 'like', '%' . $search . '%')->get();

            if (!$result->count()) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        };
    }
}
