<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Services\ReportGraphExport;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class FileController extends Controller
{
    /**
     * Download purchase file to drive
     *
     * @param mixed $file
     *
     * @return [type]
     */
    public function download($file)
    {
        // Normalize file name to prevent directory traversal
        $file = sanitizeFilename($file);

        // Define the storage path
        $filePath = "purchase/{$file}";

        // Allowed file extensions
        $allowedExtensions = ['pdf', 'jpg', 'png', 'xlsx', 'csv'];

        // Extract file extension and validate
        $extension = getFileExtension($file);

        // Check if file exists and has a valid extension
        if (!Storage::disk('public')->exists($filePath) || !in_array($extension, $allowedExtensions, true)) {
            return response()->json(['error' => 'File not found or not allowed.'], 404);
        }

        // Return the safe file download response
        return Storage::disk('public')->download($filePath);
    }

    /**
     * Save an uploaded image securely.
     *
     * This method validates and stores an image in the `public/charts` directory while preventing
     * directory traversal attacks and ensuring safe file names.
     *
     * @return JsonResponse Returns a JSON response with the stored file path or an error message.
     */
    public function saveImage()
    {
        // Validate the request (Allow JPG, JPEG, PNG up to 5MB).
        request()->validate([
            'image' => 'required|image|mimes:jpg,jpeg,png|max:5120',
            'name'  => 'required|string|max:255'
        ]);

        // Get the uploaded file.
        $image = request()->file('image');

        // if (strpos($name, '"') !== false) {
        // 	$name = str_replace('"', '&' . htmlspecialchars("quote;"), $name);
        // }

        // Sanitize filename: Allow only alphanumeric, dashes, and underscores.
        $name = preg_replace('/[^a-zA-Z0-9_-]/', '', str_replace(' ', '_', request('name')));

        // Prevent empty names after sanitization.
        if (empty($name)) {
            return response()->json(['error' => 'Invalid file name'], 400);
        }

        // Use Laravel's Storage facade to prevent direct path manipulation.
        $path = $image->storeAs('public/charts', "{$name}.jpg");

        return response()->json(['path' => $path], 201);
    }

    /**
     * Exports a graph as an Excel file.
     *
     * This function validates request parameters, sanitizes input values, 
     * ensures the provided image path is secure, and prevents directory traversal.
     * It generates an Excel file from the image and then securely deletes the image file.
     *
     * @throws \Illuminate\Validation\ValidationException If validation fails.
     * @throws \Exception If an error occurs during the export process.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse|\Illuminate\Http\RedirectResponse
     */
    public function graphExport()
    {
        $redirectUrls = [
            'Consumption Trends Report' => 'consumption-trends'
        ];

        try {
            // Validate request parameters
            request()->validate([
                'name'      => 'required|string|max:255',
                'sheettitle' => 'nullable|string|max:255',
                'image'     => 'required|string',
                'filename'  => 'required|string|max:255'
            ]);

            // Sanitize the "name" and "sheettitle" to allow only alphanumeric, underscores, and dashes.
            $name = preg_replace('/[^a-zA-Z0-9_-]/', '', str_replace(' ', '_', trim(request('name'))));
            $sheettitle = preg_replace('/[^a-zA-Z0-9_-]/', '', str_replace(' ', '_', trim(request('sheettitle', $name))));

            // Ensure the image path is safe (Prevent directory traversal attacks).
            $imagePath = request('image');

            // Verify the image file exists before proceeding.
            if (!Storage::exists($imagePath)) {
                return redirect()->back()->with('error', 'Image file not found');
            }

            // Get the full path of the stored image.
            $imageFullPath = storage_path("app/{$imagePath}");

            if (strpos($imagePath, '..') !== false || !str_starts_with($imagePath, 'public/charts/')) {
                return redirect()->back()->with('error', 'Invalid image path');
            } else {
                // Verify the image file is a valid jpg image.
                $mimeType = Storage::mimeType($imagePath);
                $extension = File::extension($imageFullPath);

                $allowedExtensions = ['jpg'];
                $allowedMimeTypes = ['image/jpeg', 'image/png'];

                // Check if extension and MIME type are valid
                if (!in_array($extension, $allowedExtensions) || !in_array($mimeType, $allowedMimeTypes)) {
                    return redirect()->back()->with('error', 'Invalid image file');
                }
            }

            $excel = Excel::download(new ReportGraphExport($name, $imageFullPath, $sheettitle ?? $name), request('filename') . '.xlsx');

            // Securely delete the image after exporting.
            Storage::delete($imagePath);

            return $excel;
        } catch (Exception $e) {
            $redirectUrl = $redirectUrls[request('name')] ?? 'assets-created-report';

            return redirect()->route($redirectUrl)->with('error', 'Some error occured. Please try again later');
        }
    }

    /**
     * Downloads the end of file.
     *
     * @param string $file The file to be downloaded.
     * @throws \Exception If the file is not found.
     * @return \Illuminate\Http\Response The response containing the file download.
     */
    public function downloadEndOfFile($file)
    {
        // Sanitize file name (strictly allow only valid PDF filenames).
        if (!preg_match('/^[a-zA-Z0-9_\-\s\(\)]+\.pdf$/', $file)) {
            Log::warning("Invalid PDF file request: " . $file);
            abort(403, "Invalid file request.");
        }

        $headers = array(
            'Content-Type: application/pdf',
        );

        $file = str_replace("-", "/", $file);
        $filePath = storage_path('app/public/' . $file);
        // Check if the file exists

        if (file_exists($filePath)) {
            return response()->download($filePath, 'EndOfLife.pdf', $headers);
        } else {
            // File not found, return a error response
            return redirect()->back()->with('error', 'EOL File not found.');
        }
    }

    /**
     * Download the file by AssetId.
     *
     * @param int $assetId The id 
     *
     * @return \Illuminate\Http\Response The response containing the file download.
     */
    public function downloadEOLFile($assetId)
    {
        $headers = array(
            'Content-Type: application/pdf',
        );

        $file = Asset::where('id', $assetId)->value('ewaste_certificate');
        $filePath = storage_path('app/public/' . $file);

        // Check if the file exists
        if (File::exists($filePath)) {
            return response()->download($filePath, 'EndOfLife.pdf', $headers);
        } else {
            // File not found, return a error response
            return redirect()->back()->with('error', 'File not found.');
        }
    }
}
