<?php

namespace App\Http\Controllers;

use App\Http\Responses\DataTableJsonResponse;
use Illuminate\Http\Request;
use App\Services\Employee\Landing;
use Illuminate\Support\Facades\Auth;
use App\Models\Asset;
use App\Services\Integrations\Tickets\TicketManagementService;
use App\User;

class EmployeeLandingController extends Controller
{
    protected $employeeLandingService;
    protected $ticketManagementService;

    public function __construct(Landing $employeeLandingService, TicketManagementService $ticketManagementService)
    {
        $this->employeeLandingService = $employeeLandingService;
        $this->ticketManagementService = $ticketManagementService;
    }

    public function index()
    {
        $userData = $this->employeeLandingService->getEmployeeData(Auth::id());

        return view('employee.landing', compact('userData'));
    }

    public function getAssets(Request $request)
    {
        $assets = $this->employeeLandingService->getEmployeeAssets($request->input('id'))->get();

        return response()->json($assets);
    }

    public function sendDispute(Request $request)
    {
        $request->validate([
            'dispute-details' => 'required',
        ]);

        $asset = Asset::with('makeAndModel')->find($request->input('asset_id'));

        if ($asset) {
            $this->employeeLandingService->sendDisputeNotification(Auth::id(), $request->input('dispute-details'), $asset);

            $user = User::find(Auth::id());
            $userName = $user->first_name . " " . $user->last_name;
            $ticketInegrationClassName = config('ticket-integration.service', null);
            $lineBreak = config('ticket-integration.' . $ticketInegrationClassName . '.newline_character') ?? '/n';
            $description = "Asset(s) reported by user as not currently being in possession/used " . $lineBreak;

            $description = $description . __('jira.DisputeCommon', [
                'asset_type'    => $asset->assetType->name,
                'make_and_model' => $asset->makeAndModel->name,
                'asset_tag' => $asset->asset_tag,
                'serial_no' => $asset->serial_no,
            ]);

            $description .= $request->input('dispute-details');
            $ticketTitleField = config('ticket-integration.' . $ticketInegrationClassName . '.ticket_title_field');
            $ticketDescriptionField = config('ticket-integration.' . $ticketInegrationClassName . '.ticket_description_field');

            $data = [
                $ticketTitleField => 'Asset Dispute - ' . $userName,
                $ticketDescriptionField => $description
            ];

            $this->ticketManagementService->createTicket($data);
        }

        return redirect('/my-teq-assets')->with('message', __('message.dispute_send'));
    }

    public function sendDisputeCommon(Request $request)
    {
        $this->employeeLandingService->sendCommonDisputeNotification(Auth::id(), $request->all());

        $data = $this->getData($request);
        $this->ticketManagementService->createTicket($data);

        return redirect('/my-teq-assets')->with('message', __('message.dispute_send'));
    }

    public function exportActiveTeam()
    {
        return $this->employeeLandingService->exportActiveTeam(Auth::id());
    }

    public function exportInActiveTeam()
    {
        return $this->employeeLandingService->exportInActiveTeam(Auth::id());
    }

    public function getData($request)
    {
        $ticketInegrationClassName = config('ticket-integration.service', null);
        $lineBreak = config('ticket-integration.' . $ticketInegrationClassName . '.newline_character') ?? '/n';
        $description = "Asset(s) reported by user as not currently being in possession/used " . $lineBreak;
        $user = User::find(Auth::id());
        $userName = $user->first_name . " " . $user->last_name;

        $jiraDescription = '';

        for ($i = 1; $i <= $request->input('asset_count'); $i++) {
            $jiraDescription .= __('jira.DisputeCommon', [
                'asset_type'    => '',
                'make_and_model' => $request->input('make_and_model' . $i),
                'asset_tag' => $request->input('asset_tag' . $i),
                'serial_no' => $request->input('serial_no' . $i)
            ]);
        }

        $description .= $jiraDescription;
        $ticketTitleField = config('ticket-integration.' . $ticketInegrationClassName . '.ticket_title_field');
        $ticketDescriptionField = config('ticket-integration.' . $ticketInegrationClassName . '.ticket_description_field');

        return [
            $ticketTitleField => 'Asset Dispute - ' . $userName,
            $ticketDescriptionField => $description
        ];
    }


    public function activeData(Request $request)
    {
        $filteredData   = $this->employeeLandingService->activeFilter();
        $users          = $filteredData['users'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($users)) {
            $data = $this->employeeLandingService->getUserData($users, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    public function inactiveData(Request $request)
    {
        $filteredData   = $this->employeeLandingService->inactiveFilter();
        $users          = $filteredData['users'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($users)) {
            $data = $this->employeeLandingService->getUserData($users, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }
}
