<?php

namespace App\Http\Controllers\DiscoveryTools\Zoom;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ZoomDevice;
use App\Services\DiscoveryTools\Zoom\ZoomService;

class IndexController extends Controller
{
    /**
     * Constructor for the IndexController class.
     *
     * @param ZoomService $serviceClass The ZoomService instance used for handling Zoom-related operations.
     */
    public function __construct(protected ZoomService $serviceClass) {}

    /**
     * Fetches filtered and formatted location records based on a search term.
     *
     * This method handles the AJAX request to fetch location data using the search term from the request.
     * The method returns a JSON response containing the filtered and formatted location records. If no results
     * are found, it returns a default record with 'id' = 0 and 'label' = 'No result found'.
     *
     * @param Request $request The request object.
     *
     * @return \Illuminate\Http\JsonResponse A JSON response containing the filtered and formatted location records.
     */
    public function getFilterFields(Request $request)
    {
        if (!empty($request->search)) {

            $result = $this->serviceClass->getFilterFields($request->search, $request->type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * Retrieve the issue detail for a given Zoom ID.
     *
     * This method takes a Zoom ID, fetches the associated issue, and returns the issue as an unordered list.
     *
     * @param int $id The Zoom ID.
     *
     * @return string The issue detail as an unordered list.
     */
    public function getIssueDetail($id)
    {
        $zoom = ZoomDevice::findOrFail($id);
        $data = '';

        if ($zoom) {
            $issue = explode(";", $zoom->issue);
            $data = '<ul>';
            foreach ($issue as $value) {
                $data .= '<li>' . $value . '</li>';
            }
            $data .= '</ul>';
        }

        return $data;
    }

    /**
     * Retrieve the AV detail for a given Zoom ID.
     *
     * This method takes a Zoom ID, fetches the associated AV data, and returns the data as an unordered list.
     *
     * @param int $id The Zoom ID.
     *
     * @return string The AV detail as an unordered list.
     */
    public function getAvDetail($id)
    {
        $zoom = ZoomDevice::findOrFail($id);
        $data = '';

        if ($zoom) {
            $data = "<ul>
            <li><label>Camera</label></li>
            <label style='font-weight: 400'>" . $zoom->camera . "</label><br>
            <li><label>Microphone</label></li>
            <label style='font-weight: 400'>" . $zoom->microphone . "</label><br>
            <li><label>Speaker</label></li>
            <label style='font-weight: 400'>" . $zoom->speaker . "</label><br>
            </ul>";
        }

        return $data;
    }
}
