<?php

namespace App\Http\Controllers\DiscoveryTools\WorkspaceOne;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use Illuminate\Http\Request;
use App\Services\DiscoveryTools\WorkspaceOne\NotCheckingInNotManaged;

class NotCheckingInNotManagedController extends Controller
{

    /**
     * Constructor for the SomeService class.
     *
     * @param NotCheckingInNotManaged $workspaceService Service for managing devices not checking in and not managed.
     */
    public function __construct(protected NotCheckingInNotManaged $workspaceService)
    {
    }

    /**
     * Showing the assets that are Deployed - Active, not Managed
     *
     * @return \Illuminate\View\View The view for the not-checking-in-not-managed page.
     */
    public function index()
    {
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.chromebook'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.workspace-one.not-checking-in-not-managed', compact('lastSyncDate'));
    }

    /**
     * Taking the data to the `Deployed - Active, not Managed` report data table
     *
     * @param Request $request The incoming request instance.
     *
     * @return ReportJsonResponse JSON response with the filtered data for the report.
     */
    public function data(Request $request)
    {
        $filteredData = $this->workspaceService->data();

        if (!empty($filteredData['assets'])) {
            $data = $this->workspaceService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export data of devices that are active but not managed to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse The response containing the CSV file.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $assetsNotCreated = $this->workspaceService->exportData();
        $requiredColumns = $requiredColumns = config('discovery-tools-export-columns.workspaceOne.DeviceActiveButNotManaged');

        $assetsNotCreated->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->workspaceService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
