<?php

namespace App\Http\Controllers\DiscoveryTools\WorkspaceOne;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use App\Services\DiscoveryTools\WorkspaceOne\AllWorkspaceOneService;
use Illuminate\Http\Request;

class AllWorkspaceOneController extends Controller
{

    /**
     * Constructor for the SomeService class.
     *
     * @param AllWorkspaceOneService $workspaceOneService Service for managing all Workspace One reports.
     */
    public function __construct(protected AllWorkspaceOneService $workspaceOneService)
    {
    }

    /**
     * Display the page showing the Workspace One report.
     *
     * @return \Illuminate\View\View The view for the Workspace One report page.
     */
    public function index()
    {
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.chromebook'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.workspace-one.all-workspace-one-report', compact('lastSyncDate'));
    }

    /**
     * Retrieve and return filtered data for the report.
     *
     * @param Request $request The incoming request instance.
     *
     * @return ReportJsonResponse JSON response with the filtered data for the report.
     */
    public function data(Request $request)
    {
        $filteredData = $this->workspaceOneService->data();

        if (!empty($filteredData['assets'])) {
            $data = $this->workspaceOneService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse The response containing the CSV file.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $allReport = $this->workspaceOneService->exportData();
        $requiredColumns = config('discovery-tools-export-columns.workspaceOne.Common');

        $allReport->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->workspaceOneService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
