<?php

namespace App\Http\Controllers\DiscoveryTools\MobileIron;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Http\Traits\DiscoveryToolsAssetUpdateTrait;
use App\Models\AssetStatus;
use App\Models\IntegrationSyncHistory;
use Illuminate\Http\Request;
use App\Services\DiscoveryTools\MobileIron\CheckingInNotCreated;
use App\Models\MobileIron;

/**
 * Controller class for Devices Not Created report
 */
class CheckingInNotCreatedController extends Controller
{
    use DiscoveryToolsAssetUpdateTrait;

    public function __construct(CheckingInNotCreated $mobileIronService)
    {
        $this->service = $mobileIronService;
    }

    /**
     * Showing the Devices that are not in teqtivity but in MobileIron
     * @return Http\Response
     */
    public function index()
    {
        $assetStatuses = AssetStatus::orderBy('slug')->get();
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.mobile-iron'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.mobile-iron.checking-in-not-created', compact('assetStatuses', 'lastSyncDate'));
    }

    /**
     * Taking the data to the `Device Backup` report
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $filteredData = $this->service->data();
        $assets = $filteredData['assets'];


        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, $start, $data);
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Taking the data to the `Devices backup` report
     *
     * @return Http\Response
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();
        $filteredData = $this->service->exportData();
        $requiredColumns = config('discovery-tools-export-columns.MobileIron.Common');
        $filteredData->chunk(5000, function ($filteredDataChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->service->getReportExportData($filteredDataChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }

    /**
     * create
     *
     * @return void
     */
    public function create()
    {
        $mobileIronAsset = MobileIron::find(request('mobile-iron-asset-id'));
        $this->CreateAsset($mobileIronAsset);

        return redirect('/discovery-tools/mobileiron/checking-in-not-created')->with('message', 'Asset created successfully.');
    }
}
