<?php

namespace App\Http\Controllers\DiscoveryTools\Kandji;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\KandjiDevices;
use Illuminate\Http\Request;
use App\Services\DiscoveryTools\Kandji\CheckingInNotCreated;
use App\Http\Traits\DiscoveryToolsAssetUpdateTrait;
use App\Models\AssetStatus;
use App\Models\IntegrationSyncHistory;

/**
 * Controller class for Devices Not Created report
 */
class CheckingInNotCreatedController extends Controller
{

    use DiscoveryToolsAssetUpdateTrait;

    public function __construct(CheckingInNotCreated $service)
    {
        $this->service = $service;
    }

    /**
     * Showing the Devices that are not in teqtivity but in Kandji
     * @return Http\Response
     */
    public function index()
    {
        $assetStatuses = AssetStatus::orderBy('slug')->get();
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.kandji'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.kandji.checking-in-not-created', compact('assetStatuses', 'lastSyncDate'));
    }

    /**
     * Taking the data to the `Device Backup` report
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $filteredData = $this->service->data();
        $assets = $filteredData['assets'];


        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, $start, $data);
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }


    /**
     * Taking the data to the `Devices backup` report
     *
     * @return Http\Response
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();
        $filteredData = $this->service->exportData();
        $requiredColumns = config('discovery-tools-export-columns.Kandji.Common');
        $filteredData->chunk(5000, function ($filteredDataChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->service->getReportExportData($filteredDataChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }

    /**
     * create
     *
     * @return void
     */
    public function create()
    {
        $kandjiAsset = KandjiDevices::find(request('kandji-asset-id'));
        $this->CreateAsset($kandjiAsset);
        return redirect('/discovery-tools/kandji/checking-in-not-created')->with('message', 'Asset created successfully.');
    }
}
