<?php

namespace App\Http\Controllers\DiscoveryTools\Jamf;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Http\Traits\DiscoveryToolsAssetUpdateTrait;
use App\Models\AssetStatus;
use App\Models\IntegrationSyncHistory;
use App\Services\DiscoveryTools\Jamf\PreStageDevicesService;

class PreStageDevicesController extends Controller
{
    use DiscoveryToolsAssetUpdateTrait;
    protected $jamfService;

    public function __construct(PreStageDevicesService $jamfService)
    {
        $this->jamfService = $jamfService;
    }

    /**
     * Showing the 'All Jamf' Report
     * @return view
     */
    public function index()
    {
        $assetStatuses = AssetStatus::orderBy('slug')->get();
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.jamf'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.jamf.pre-stages', compact('assetStatuses', 'lastSyncDate'));
    }

    /**
     * Taking the data to the `All Jamf` report
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        $filteredData = $this->jamfService->data();

        if (!empty($filteredData['assets'])) {
            $data = $this->jamfService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export data to CSV format.
     *
     * @return string
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();

        $allJamfDevices =  $this->jamfService->exportData();
        $requiredColumns = config('discovery-tools-export-columns.jamf.Common');

        $allJamfDevices->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->jamfService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
