<?php

namespace App\Http\Controllers\DiscoveryTools\Jamf;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Services\DiscoveryTools\Jamf\CheckingInOnHand;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use Cache;

class CheckingInOnHandController extends Controller
{

    /**
     * Initialize the controller with the CheckingInOnHand service.
     *
     * @param \App\Services\CheckingInOnHand $jamfService The service to manage checking in on-hand assets
     */
    public function __construct(protected CheckingInOnHand $jamfService)
    {
    }

    /**
     * Display the Checking In On-Hand view.
     *
     * @return \Illuminate\View\View The view for checking in on-hand assets
     */
    public function index()
    {
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.jamf'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.jamf.checking-in-on-hand', compact('lastSyncDate'));
    }

    /**
     * Retrieve and format the data for checking in on-hand assets.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the formatted data for the report
     */
    public function data(Request $request)
    {
        $filteredData = $this->jamfService->data();

        if (!empty($filteredData['assets'])) {
            $data = $this->jamfService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export data for checking in on-hand assets to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response containing the CSV file for download
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $assetsNotCreated = $this->jamfService->exportData();
        $requiredColumns = config('discovery-tools-export-columns.jamf.Common');

        $assetsNotCreated->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->jamfService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
