<?php

namespace App\Http\Controllers\DiscoveryTools\Jamf;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use App\Services\DiscoveryTools\Jamf\AllJamfService;

class AllJamfController extends Controller
{
    protected $jamfService;

    public function __construct(AllJamfService $jamfService)
    {
        $this->jamfService = $jamfService;
    }

    /**
     * Showing the 'All Jamf' Report
     * @return view
     */
    public function index()
    {
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.intune'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.jamf.all-report', compact('lastSyncDate'));
    }

    /**
     * Taking the data to the `All Jamf` report
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        $filteredData = $this->jamfService->data();

        if (!empty($filteredData['assets'])) {
            $data = $this->jamfService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Export data for 'All Jamf' report to a CSV file.
     *
     * This function retrieves data from the Jamf service, processes it in chunks,
     * and exports it to a CSV format. The export includes columns specified in
     * the configuration for Jamf discovery tools.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response containing the CSV file for download
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();

        $allJamfDevices =  $this->jamfService->exportData();
        $requiredColumns = config('discovery-tools-export-columns.jamf.Common');

        $allJamfDevices->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->jamfService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
