<?php

namespace App\Http\Controllers\DiscoveryTools\Intune;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use App\Services\DiscoveryTools\Intune\AllIntuneService;

class AllIntuneController extends Controller
{
    protected $intuneService;

    public function __construct(AllIntuneService $intuneService)
    {
        $this->intuneService = $intuneService;
    }

    /**
     * Showing the 'All Intune' Report
     * @return view
     */
    public function index()
    {
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.intune'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('discovery-tools.intune.all-intune-report', compact('lastSyncDate'));
    }

    /**
     * Taking the data to the `All Intune` report
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        $filteredData = $this->intuneService->data();
        $data = [];
        if (!empty($filteredData['assets'])) {
            $data = $this->intuneService->getReportData($filteredData['assets'], request('start'));
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * Exporting the `All Intune` report
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();
        $assetsNotCreated =  $this->intuneService->exportData();
        $requiredColumns = config('discovery-tools-export-columns.intune.Common');

        $assetsNotCreated->chunk(5000, function ($assetChunks) use ($data, $datas, $requiredColumns) {
            $datas->push($this->intuneService->getReportExportData($assetChunks, 0, $data, $requiredColumns));
        });

        return exportToCsv($datas->toArray(), $requiredColumns);
    }
}
