<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\ActiveUsersCountInMonth;
use App\Services\Dashboard\HrDashboardService;
use Carbon\Carbon;
use Artisan;
use Illuminate\Http\Request;

/**
 * Controller class that handles the functionality and operations related to the HR Dashboard.
 */
class HrDashboardController extends Controller
{

    /**
     * Initialize the controller with the HrDashboardService.
     *
     * @param \App\Services\HrDashboardService $hrDashboardService The service to manage HR dashboard operations
     */
    public function __construct(protected HrDashboardService $hrDashboardService) {}

    /**
     * Display the HR dashboard index view.
     *
     * @return \Illuminate\View\View The view for the HR dashboard index
     */
    public function index()
    {
        $totalUsers = ActiveUsersCountInMonth::get()->toArray();

        return view('dashboard.hr-dashboard', compact('totalUsers'));
    }

    /**
     * Get the onboarding summary for a specified date range.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the onboarding summary data
     */
    public function onboardingSummary(Request $request)
    {
        $fromDate = request('form')['from_date'] ? Carbon::createFromFormat(config('date.formats.month_year_format'), request('form')['from_date'])->startOfMonth()->format(config('date.formats.read_date_format')) : Carbon::now()->startOfMonth()->format(config('date.formats.read_date_format'));
        $toDate = request('form')['to_date'] ? Carbon::createFromFormat(config('date.formats.month_year_format'), request('form')['to_date'])->endOfMonth()->format(config('date.formats.read_date_format')) : "";
        $data = $this->hrDashboardService->getOnboardingSummary($fromDate, $toDate);
        $totalData = 2;

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Get the termination summary for a specified date range.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the termination summary data
     */
    public function terminationSummary(Request $request)
    {
        $fromDate   = request('form')['from_date'] ? Carbon::createFromFormat(config('date.formats.month_year_format'), request('form')['from_date'])->startOfMonth()->format(config('date.formats.read_date_format')) : "";
        $toDate     = request('form')['to_date'] ? Carbon::createFromFormat(config('date.formats.month_year_format'), request('form')['to_date'])->endOfMonth()->format(config('date.formats.read_date_format')) : "";
        $data       = $this->hrDashboardService->getTerminationSummary($fromDate, $toDate);
        $totalData  = 2;

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Refresh the active users count for the last twelve months.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message
     */
    public function refreshActiveUsersCount()
    {
        Artisan::call('ActiveUserCountInLastTwelveMonths:update');

        return redirect('/hr-dashboard')->with('message', 'Active users count updated successfully');
    }
}
