<?php

namespace App\Http\Controllers\CloudAssets;

use App\Http\Controllers\Controller;
use App\Http\Responses\ReportJsonResponse;
use App\Models\IntegrationSyncHistory;
use App\Services\CloudAssets\CloudAssetService;
use Exception;

class CloudAssetsController extends Controller
{
    protected $service;

    /**
     * Constructor for the class.
     *
     * @param CloudAssetService $service
     */
    public function __construct(CloudAssetService $service)
    {
        $this->service = $service;
    }

    /**
     * Load view file
     *
     * @return View
     */
    public function index()
    {
        $cloudAssetTypes = config('aws-cloud-assets.cloud_asset_type');
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.cloud-assets'))->first();
        $lastSyncDate = $lastSync?->created_at;
        $regions = config('aws-cloud-assets.regions');

        return view('cloud-assets.index', compact('lastSyncDate', 'cloudAssetTypes', 'regions'));
    }
    /**
     * Taking data for Cloud Assets
     *
     * @return Illuminate\Http\JsonResponse
     */
    public function data()
    {
        try {
            $filteredData = $this->service->data();
            $assets =  $filteredData['cloudAssets'];
            $data = [];

            if (!empty($assets)) {
                $data = $this->service->getReportData($assets, request('start'));
            }

            return new ReportJsonResponse(request()->input('draw'), $data, $filteredData['count']);
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage()], 500);
        }

    }

    /**
     * Taking the data to the report export
     *
     * @return Http\Response
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $dataCollection = collect();
        $assets = $this->service->exportData();
        $assets->chunk(5000, function ($assetChunks) use ($data, $dataCollection) {
            $dataCollection->push($this->service->getReportExportData($assetChunks, $data));
        });

        return exportToCsv($dataCollection->toArray());
    }

    /**
     * Filter field values by field name
     */
    public function filterFieldValues()
    {
        try {
            $result = $this->service->getFilterFieldValues();

            if (!count($result)) {
                $result[] = ['value' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }
}