<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\User;
use Illuminate\Support\Facades\Auth;
use Socialite;
use SocialiteProviders\Manager\Config;
use App\Models\ApiCredential;
use Illuminate\Support\Facades\Log;

class OktaLoginController extends Controller
{
    protected $config;

    public function __construct()
    {
        $credential = ApiCredential::where('slug', 'okta_sso')->first();

        $clientId = optional($credential)->user_name;
        $clientSecret = $credential ? optional($credential)->password : '';
        $redirectUrl = url('login/okta/callback');
        $baseUrl = optional($credential)->url;

        $this->config = new Config($clientId, $clientSecret, $redirectUrl, ['base_url' => $baseUrl]);
    }

    /**
     * Redirect the user to the Okta authentication page.
     *
     * @return \Illuminate\Http\Response
     */
    public function redirectToProvider()
    {
        return Socialite::with('okta')->setConfig($this->config)->redirect();
    }

    /**
     * Obtain the user information from Okta.
     *
     * @return \Illuminate\Http\Response
     */
    public function handleProviderCallback(Request $request)
    {
        try {
            $user = Socialite::with('okta')->setConfig($this->config)->user();
        } catch (\Throwable $e) {
            Log::channel('daily')->info("Octa login error ---- " . $e);
            return redirect('/login')->withErrors([
                'saml2_error_single' => 'The authorization code is invalid or has expired.'
            ]);
        }

        $localUser = User::where('email', $user->email)->first();

        if ($localUser) {
            $localUser->token = $user->token;
            $localUser->save();
        }

        try {
            Auth::login($localUser);
        } catch (\Throwable $e) {
            return redirect('/login')->withErrors([
                'saml2_error_single' => 'You do not have access'
            ]);
        }

        return $this->authenticated($request, $localUser);
    }

    public function authenticated(Request $request, $user)
    {
        if ($user->userType->name == 'Employee') {

            return redirect('my-teq-assets');
        }

        if ($user->hasPermissionTo('IT Assets View')) {
            return redirect('/assets');
        }

        if ($user->hasPermissionTo('Research Assets View')) {
            return redirect('/research-assets');
        }

        return redirect('/');
    }

    public function logout()
    {
        Auth::logout();
        return redirect('/');
    }
}
