<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\SendsPasswordResetEmails;
use App\Events\ResetPasswordEmailFailed;
use App\Http\Traits\ResetPasswordFailedAttempt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Password;

class ForgotPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset emails and
    | includes a trait which assists in sending these notifications from
    | your application to your users. Feel free to explore this trait.
    |
    */

    use SendsPasswordResetEmails, ResetPasswordFailedAttempt;

    /**
     * Lock if someone gets their password wrong 10 times for 10 minutes
     */
    protected $maxAttempts = 5;
    protected $decayMinutes = 2;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(Request $request)
    {
        $this->middleware('guest');
    }

    /**
     * Send a reset link to the given user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function sendResetLinkEmail(Request $request)
    {
        $this->validateEmail($request);

        //ip lockout on many forgot password attempts

        if ($this->hasTooManyResetPasswordAttempts($request)) {
            $this->fireLockoutEvent($request);

            // Log the lockout event
            Log::channel('daily')->info('Forgot password lockout from IP ' . \Request::ip() . ' for Email ' . $request->email);

            return $this->sendLockoutResponse($request);
        }

        $this->incrementResetAttempts($request);

        // We will send the password reset link to this user. Once we have attempted
        // to send the link, we will examine the response then see the message we
        // need to show to the user. Finally, we'll send out a proper response.
        $response = $this->broker()->sendResetLink(
            $this->credentials($request)
        );

        return $response == Password::RESET_LINK_SENT
            ? $this->sendResetLinkResponse($request, $response)
            : $this->sendResetLinkFailedResponse($request, $response);
    }

    // Specify the username field here  we have email so for the throttle functions we need to specify the username field
    public function username()
    {
        return 'email';
    }
}
