<?php

namespace App\Http\Controllers\AssetsHealth;

use App\Http\Controllers\Controller;
use App\Models\AssetHealthHistory;
use App\Models\AssetHealthReportSchedule;
use App\Services\AssetsHealth\AssetsHealthExportService;

class AssetsHealthReportExportController extends Controller
{
    /**
     * Constructor for the class.
     *
     * @param AssetsHealthExportService $assetsHealthExportService description
     */
    public function __construct(protected AssetsHealthExportService $assetsHealthExportService)
    {
    }

    /**
     * Display the index page for exporting asset health reports.
     *
     * @return \Illuminate\Contracts\View\View The view for exporting asset health reports.
     */
    public function index()
    {
        $schedule = AssetHealthReportSchedule::first();

        $latestHistory = AssetHealthHistory::latest()->first();

        return view('assets-health.export-report', compact('schedule', 'latestHistory'));
    }

    /**
     * Download the asset health test reports.
     *
     * @return \Illuminate\Http\Response The response containing the downloaded file.
     */
    public function download()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $healthTestData = $this->assetsHealthExportService->getTestReportsData();

        $healthTestData->chunk(10000, function ($testDetailChunks) use ($data, $datas) {
            $datas->push($this->assetsHealthExportService->getTestReportDownloadData($testDetailChunks, 0, $data));
        });

        return $this->assetsHealthExportService->export($datas->toArray());
    }

    /**
     * Retrieve the data for generating a PDF sreport of asset health test data.
     *
     * @return \Illuminate\Http\Response The response containing the PDF report data.
     */
    public function pdfData()
    {
        return $this->assetsHealthExportService->render();
    }

    /**
     * Download the latest asset health report.
     *
     * @return \Illuminate\Http\Response The response containing the downloaded report file.
     */
    public function downloadReport()
    {
        $lastHistory = AssetHealthHistory::orderByDesc('created_at')->first();

        if (!$lastHistory) {
            return redirect()->route('asset-health-report-export')->with('error', 'No report found.');
        }

        $filePath = public_path('reports/' . $lastHistory->created_at->format('Y-m-d') . '/report.pdf');

        if (!file_exists($filePath)) {
            return redirect()->route('asset-health-report-export')->with('error', 'File not found.');
        }

        return response()->download($filePath);
    }
}
