<?php

namespace App\Http\Controllers\AssetsHealth;

use App\Http\Controllers\Controller;
use App\Models\AssetHealthTests;
use App\Services\AssetsHealth\AssetsHealthDashboardService;
use Illuminate\Http\Request;
use App\Services\AssetsHealth\AssetsHealthReportService;
use App\Services\AssetsHealth\AssetsHealthReportSyncService;
use Log;
use App\Services\Integrations\Tickets\TicketManagementService;

class AssetsHealthReportController extends Controller
{
    /**
     * Constructor for initializing dependencies.
     *
     * @param AssetsHealthDashboardService $assetHealthDashboardService Service for asset health dashboard.
     * @param AssetsHealthReportService $assetHealthReportService Service for asset health reports.
     * @param AssetsHealthReportSyncService $assetHealthReportSyncService Service for synchronizing asset health reports.
     * @param TicketManagementService $ticketManagementService Service for ticket management.
     *
     * @return void
     */
    public function __construct(
        protected AssetsHealthDashboardService $assetHealthDashboardService,
        protected AssetsHealthReportService $assetHealthReportService,
        protected AssetsHealthReportSyncService $assetHealthReportSyncService,
        protected TicketManagementService $ticketManagementService
    ) {
    }

    /**
     * Retrieves the asset health reports for a given category slug.
     *
     * @param string|null $categorySlug The slug of the category. If null, retrieves reports for all categories.
     *
     * @return \Illuminate\Contracts\View\View The view that displays the asset health reports.
     */
    public function index($categorySlug = null)
    {
        $testReports = [];

        $allCategories = $this->assetHealthDashboardService->getCategories();

        foreach ($allCategories as $category) {
            $testReports[$category->slug] = $this->assetHealthDashboardService->getHealthReportsByTests($category->id)->get();
        }

        return view('assets-health.reports', compact('categorySlug', 'testReports', 'allCategories'));
    }

    /**
     * Perform manual synchronization of asset health test reports and create a PDF report.
     *
     * @return \Illuminate\Http\JsonResponse JSON response indicating the completion status and count of synchronized reports.
     */
    public function sync()
    {
        $healthTestReports = $this->assetHealthReportSyncService->sync();

        $this->assetHealthReportSyncService->createReportPdf();

        $responseData = [
            'message' => 'Manual Sync Completed successfully.',
            'count' => $healthTestReports['count']
        ];

        return response()->json($responseData);
    }

    /**
     * Display the form for creating a new ticket.
     *
     * @return \Illuminate\Contracts\View\View The view for creating a new ticket.
     */
    public function createTicket()
    {
        return view('assets-health.create-ticket');
    }

    /**
     * Create a new Jira ticket based on the provided request data.
     *
     * @param  \Illuminate\Http\Request  $request The HTTP request containing ticket data.
     *
     * @return \Illuminate\Http\RedirectResponse  A redirect back to the previous page with a success or error message.
     */
    public function createJiraTicket(Request $request)
    {
        if ($request->filled('description')) {
            $data = [
                'summary' => $request->input('summary'),
                'description' => $request->input('description')
            ];

            $this->createTicketCall($data);

            return back()->with('message', 'Ticket created successfully');
        }

        return back()->with('error', 'Please enter description');
    }

    /**
     * Create a Jira ticket using the provided ticket data.
     *
     * @param  array  $ticketData  The data for creating the ticket (summary, description).
     *
     * @return mixed               The ID of the created ticket if successful, otherwise null.
     */
    private function createTicketCall(array $ticketData)
    {
        try {
            $issue = $this->ticketManagementService->createTicket($ticketData);

            return !empty($issue->id) ? $issue->id : null;
        } catch (\Exception $e) {
            Log::channel('daily')->info('Jira Ticket Create Error :  ' . $e->getMessage());
        }

        return true;
    }

    /**
     * Disable an asset health test.
     *
     * @param  int  $assetHealthTestId  The ID of the asset health test to disable.
     *
     * @return \Illuminate\Http\RedirectResponse  A redirect back with a success message.
     */
    public function disableAssetHealthTest($assetHealthTestId)
    {
        $assetHealthTest = AssetHealthTests::find($assetHealthTestId);
        $assetHealthTest->status = 0;
        $assetHealthTest->save();

        return redirect()->route('asset-health-test-detail', ['asset_health_report_id' => $assetHealthTest->asset_health_report_id])->with('success', 'Asset Health Test disabled successfully.');
    }

    /**
     * Enable an asset health test.
     *
     * @param  int  $assetHealthTestId  The ID of the asset health test to enable.
     *
     * @return \Illuminate\Http\RedirectResponse  A redirect back with a success message.
     */
    public function enableAssetHealthTest($assetHealthTestId)
    {
        $assetHealthTest = AssetHealthTests::find($assetHealthTestId);
        $assetHealthTest->status = 1;
        $assetHealthTest->save();

        return redirect()->route('asset-health-test-detail', ['asset_health_report_id' => $assetHealthTest->asset_health_report_id])->with('success', 'Asset Health Test enabled successfully.');
    }
}
