<?php

namespace App\Http\Controllers\AssetsHealth;

use App\Http\Controllers\Controller;
use App\Services\AssetsHealth\AssetsHealthDashboardService;
use App\Services\AssetsHealth\AssetsHealthHistoryService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class AssetsHealthHistoryController extends Controller
{


    /**
     * Create a new controller instance.
     *
     * @param  \App\Services\AssetsHealthHistoryService     $assetsHealthHistoryService     The service for managing assets health history.
     * @param  \App\Services\AssetsHealthDashboardService  $assetsHealthDashboardService  The service for managing assets health dashboard.
     */
    public function __construct(
        protected AssetsHealthHistoryService $assetsHealthHistoryService,
        protected AssetsHealthDashboardService $assetsHealthDashboardService
    ) {
    }


    /**
     * Display the index page of assets health history.
     *
     * @return \Illuminate\Contracts\View\View The view for the assets health history index page.
     */
    public function index()
    {
        $getAllCategories = $this->assetsHealthDashboardService->getCategoriesWithTests()->get();

        return view('assets-health.health-history',  compact('getAllCategories'));
    }


    /**
     * Retrieve data for assets health history.
     *
     * @param  \Illuminate\Http\Request  $request The HTTP request.
     *
     * @return \Illuminate\Http\JsonResponse        The JSON response containing assets health history data.
     */
    public function data(Request $request)
    {
        $year = carbon::now()->year;
        if ($request->has('year')) {
            $year = $request->input('year');
        }

        $testId = null;
        if ($request->has('testId')) {
            $testId = $request->input('testId');
        }

        $categoryId = null;
        if ($request->has('categoryId')) {
            $categoryId = $request->input('categoryId');
        }

        $inputData = [
            'dateFrom' => $this->assetsHealthHistoryService->getFromDate(),
            'dateTo' => $this->assetsHealthHistoryService->getToDate(),
            'testId' => $testId,
            'categoryId' => $categoryId,
        ];

        $datas = $this->assetsHealthHistoryService->assetsHealthReportData($inputData);
        $graphData = $datas['datas'];
        $weekNames = $datas['weeks'];

        $historyData = [
            'graphData' => $graphData,
            'weekNames' => $weekNames,
            'year' => $year,
        ];

        return response()->json($historyData);
    }


    /**
     * Filter and retrieve data for assets health history.
     *
     * @param  \Illuminate\Http\Request  $request The HTTP request containing filter parameters.
     *
     * @return \Illuminate\Http\JsonResponse        The JSON response containing filtered assets health history data.
     */
    public function filterData(Request $request)
    {

        $dateTo = $request->filled('to_date') ? Carbon::parse($request->to_date) : Carbon::now();
        $dateToCopy = $dateTo->copy();

        $dateFrom = $request->filled('from_date') ? Carbon::parse($request->from_date) : $dateToCopy->subDays(84)->startOfWeek();

        $inputData = [
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'testIds' => $request->get('testIds'),
            'categoryIds' => $request->get('categoryIds'),
        ];

        $datas = $this->assetsHealthHistoryService->assetsHealthReportDataFilter($inputData);

        return response()->json([
            'graphData' => $datas['datas'],
            'weekNames' => $datas['weeks'],
            'labels' => $datas['labels'],
            'ykeys' => $datas['ykeys'],
            'colorArray' => $datas['colorArray']
        ]);
    }
}
