<?php

namespace App\Http\Controllers\AssetsHealth;

use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\AssetHealthReports;
use App\Models\AssetHealthTests;
use Illuminate\Http\Request;
use App\Services\AssetsHealth\AssetsHealthDashboardService as AssetsHealthDashboardService;

class AssetsHealthController extends Controller
{


    /**
     * Constructor for the class.
     *
     * @param AssetsHealthDashboardService $assetsHealthDashboardService description
     */
    public function __construct(protected AssetsHealthDashboardService $assetsHealthDashboardService)
    {
    }


    /**
     * Display the dashboard for assets health.
     *
     * @return \Illuminate\Contracts\View\View The view for the assets health dashboard.
     */
    public function index()
    {
        $categories = $this->assetsHealthDashboardService->getCategories();
        $count = $this->assetsHealthDashboardService->getCategoryTestCount();
        $topIssues = $this->assetsHealthDashboardService->getTopAssetHealthIssues()->get();

        $assetHealthData = $this->assetsHealthDashboardService->getAssetHealthData();

        $latestHistory = $this->assetsHealthDashboardService->getLatestAssetHealthHistory();

        $assetHealth = [
            'labels' => ['Without Issues', 'With Issues'],
            'datasets' => [
                [
                    'data' => [$assetHealthData, (100 - $assetHealthData)],
                    'backgroundColor' => ['#0da5f9', '#cfccc6'],
                ],
            ],
        ];

        $scannedAssetsData = $this->assetsHealthDashboardService->getscannedAssetsData();
        $scannedAssets = [];

        foreach ($scannedAssetsData as $key => $name) {
            $scannedAssets[] = ['label' => $key, 'value' => $name];
        }

        $assetHistories = $this->assetsHealthDashboardService->getHistoriesData();

        return view('assets-health.dashboard', compact('count', 'categories', 'topIssues', 'assetHealth', 'scannedAssets', 'assetHistories', 'latestHistory'));
    }


    /**
     * Get the details of a test report by its ID.
     *
     * @param int $testReportId The ID of the test report.
     *
     * @return \Illuminate\Contracts\View\View The view for the test details.
     */
    public function getTestDetail($testReportId)
    {
        $data = $this->assetsHealthDashboardService->getTestDetailById($testReportId)->first();

        return view('assets-health.test-details', compact('data'));
    }


    /**
     * Get the details of an ignored test report by its ID.
     *
     * @param int $testReportId The ID of the ignored test report.
     *
     * @return \Illuminate\Contracts\View\View The view for the ignored test details.
     */
    public function getIgnoredTestDetail($testReportId)
    {
        $data = $this->assetsHealthDashboardService->getIgnoredTestDetailById($testReportId)->first();

        return view('assets-health.ignored-test-details', compact('data'));
    }


    /**
     * Display test detail data for a given test report ID.
     *
     * @param  \Illuminate\Http\Request  $request      The HTTP request.
     * @param  int                       $testReportId The ID of the test report.
     *
     * @return \Illuminate\Http\JsonResponse             The JSON response containing test detail data.
     */
    public function getTestDetailData(Request $request, $testReportId)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $healthTestData = $this->assetsHealthDashboardService->getTestReportDetails($testReportId);
        $testDetails    = $healthTestData['testDetails'];
        $totalData      = $healthTestData['count'];

        $start  = $request->input('start');
        $data   = [];

        $testData = AssetHealthTests::find(AssetHealthReports::find($testReportId)->test_id) ?? null;

        if (!empty($testDetails)) {
            $data = $this->assetsHealthDashboardService->getReportData($testDetails, $start, $data, $testData);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }


    /**
     * Display ignored test detail data for a given test report ID.
     *
     * @param  \Illuminate\Http\Request  $request      The HTTP request.
     * @param  int                       $testReportId The ID of the ignored test report.
     *
     * @return \Illuminate\Http\JsonResponse           The JSON response containing ignored test detail data.
     */
    public function getIgnoredTestDetailData(Request $request, $testReportId)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $healthTestData = $this->assetsHealthDashboardService->getIgnoredTestReportDetails($testReportId);
        $testDetails    = $healthTestData['testDetails'];
        $totalData      = $healthTestData['count'];

        $start  = $request->input('start');
        $data   = [];

        $testData = AssetHealthTests::find(AssetHealthReports::find($testReportId)->test_id) ?? null;

        if (!empty($testDetails)) {
            $data = $this->assetsHealthDashboardService->getIgnoredReportData($testDetails, $start, $data, $testData);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }


    /**
     * Export test detail data for a given test report ID.
     *
     * @param  \Illuminate\Http\Request  $request      The HTTP request.
     * @param  int                       $testReportId The ID of the test report.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse The streamed response containing exported test detail data.
     */
    public function getTestDetailExport(Request $request, $testReportId)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();
        $healthTestData = $this->assetsHealthDashboardService->getTestDetailExport($testReportId);

        $testData = AssetHealthTests::find(AssetHealthReports::find($testReportId)->test_id) ?? null;

        $healthTestData->chunk(20000, function ($testDetailChunks) use ($data, $datas, $testData) {
            $datas->push($this->assetsHealthDashboardService->getTestDetailExportData($testDetailChunks, 0, $data, $testData));
        });

        return $this->assetsHealthDashboardService->export($datas->toArray());
    }


    /**
     * Ignore a test detail by its report ID.
     *
     * @param  int  $testReportId The ID of the test report.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function ignoreTestDetail($testReportId)
    {
        $this->assetsHealthDashboardService->toggleIgnoreTestDetailById($testReportId, true);

        return redirect()->back()->with('message', 'The result has been marked as ignored.');
    }


    /**
     * Undo ignoring a test detail by its report ID.
     *
     * @param  int  $testReportId The ID of the test report.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function undoIgnoreTestDetail($testReportId)
    {
        $this->assetsHealthDashboardService->toggleIgnoreTestDetailById($testReportId, false);

        return redirect()->back()->with('message', 'The result has been marked as unignored.');
    }
}
