<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Http\Controllers\Controller;
use App\Services\SoftwareLicense\ImportedAssets\Zoom\ZoomService;
use App\Http\Requests\Zoom\ZoomCredentialsRequest;
use App\Http\Requests\Zoom\ZoomCredentialUpdateRequest;
use Illuminate\Support\Facades\Artisan;

class ZoomCredentialsController extends Controller
{

    /**
     * Constructor for ZoomCredentialsController
     *
     * @param ZoomService $service
     */
    public function __construct(protected ZoomService $service)
    {
    }

    /**
     * If the user has already created the Software asset API credential, show the user the settings page.
     * Otherwise, redirect the user to the create page
     *
     * @return object View is being returned.
     */
    public function index()
    {
        $credentialsData =  $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect()->route('zoom.create');
        }

        $status = $this->service->checkConnection($credentialsData);

        return view('settings.zoom.index', ['credentials' => $credentialsData, 'status' => $status]);
    }

    /**
     * create the new software asset API credential
     *
     * @return object View is being returned
     */
    public function create()
    {
        if ($this->service->getCredentials()) {

            return redirect()->route('zoom.index');
        }

        return view('settings.zoom.create');
    }

    /**
     * Store the Zoom Credentials in the API Credential table
     *
     * @param ZoomCredentialsRequest $request
     *
     * @return object Redirecting to the proper page
     */
    public function store(ZoomCredentialsRequest $request)
    {
        if ($this->service->getCredentials()) {
            return redirect()->route('zoom.index')->with('error', 'Zoom credentials already exist. ');
        }

        $response = $this->service->addCredentials($request);

        return $response ? redirect()->route('zoom.index')->with('message', __('message.created'))
            : redirect()->route('zoom.create')->with('error', 'Opertaion Failed');
    }

    /**
     * It takes the id of the credential, finds it in the database, and returns a view with the
     * credential
     *
     * @return object View is being returned.
     */
    public function edit()
    {
        $credentialsData = $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect()->route('zoom.create');
        }

        return view('settings.zoom.edit', ['credential' => $credentialsData]);
    }

    /**
     * It updates the credentials of the software asset account.
     *
     * @param ZoomCredentialUpdateRequest $request
     *
     * @return object Redirecting to the proper page
     */
    public function update(ZoomCredentialUpdateRequest $request)
    {
        $response = $this->service->updateCredentials($request);

        return $response ? redirect(route('zoom.index'))->with('message', 'Credentials updated successfully') :
            redirect(route('zoom.index'))->with('error', 'Something went wrong. Try again later');
    }

    /**
     * Establishing the connection with the added software asset account
     *
     * @return object Redirect to the proper page after connection
     */
    public function connect()
    {
        $connectionStatus = $this->service->setConnection();

        return $connectionStatus ? redirect()->route('zoom.index')->with('message', 'Connected successfully')
            : redirect()->route('zoom.index')->with('error', 'Connection attempt failed. Please check your credentials');
    }

    /**
     * A function to sync the data from software asset to our database.
     *
     * @return object Redirect to the proper page after connection
     */
    public function manualSync()
    {
        Artisan::call('Zoom:sync');
        $softwareAssetSyncStatus = Artisan::output();
        $response = json_decode($softwareAssetSyncStatus, true);

        if ($response['status'] == 'success') {
            return redirect(route('zoom.index'))->with('message', $response['message']);
        }

        return redirect(route('zoom.index'))->with('error', $response['message']);
    }

    /**
     * This function changes the integration status of software asset and enables/disables the software asset accordingly
     *
     * @return bool
     */
    public function integrationStatusChange()
    {
        return $this->service->integrationStatusChange();
    }
}
