<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\Vendor\VendorData;
use App\Http\Requests\StoreVendor;
use App\Http\Requests\UpdateVendor;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\VendorCategory;
use App\Models\Vendor;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;


class VendorController extends Controller
{
    public function __construct(protected VendorData $service) {}
    /**
     * Display the vendor listing page.
     *
     * This function retrieves all vendor categories and passes them to the view.
     * It then renders the view.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $categories = VendorCategory::select('id', 'name')->get();

        return view('settings.vendor.vendors', compact('categories'));
    }

    /**
     * Get vendor data as a JSON response.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance containing input parameters.
     *
     * @return \Illuminate\Http\JsonResponse JSON response containing the vendor data.
     */
    public function data(Request $request)
    {
        $filteredData   = $this->service->filter();
        $vendors        = $filteredData['vendors'];
        $totalData      = $filteredData['count'];
        
        $start  = request('start');
        $data   = [];

        if (!empty($vendors)) {
            $data = $this->service->getVendorData($vendors, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Store a new vendor in the database.
     *
     * This function handles the creation of a new vendor based on
     * the validated request data. It establishes category relations
     * for the vendor, logs the creation action in the asset history,
     * and triggers a bulk update event. Finally, it redirects to the 
     * vendor listing page with a success message.
     *
     * @param StoreVendor $request The incoming request containing the vendor data.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the vendor listing with a success message.
     */
    public function store(StoreVendor $request)
    {
        $data = $request->only(["name"]);
        $vendor = Vendor::create($data);
        //Add category relations
        $this->service->createVendorCategoryRelation($vendor->id, $request->categories);
        $description = __('history.VendorCreated', [
            'vendor' => $request['name'],
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'vendor_created',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return redirect(route('vendor-listing'))->with('message', 'Vendor created successfully.');
    }

    /**
     * Update an existing vendor in the database.
     *
     * This function handles the updating of an existing vendor based on
     * the validated request data. It updates the vendor, logs the action
     * in the asset history, and triggers a bulk update event. Finally, it
     * redirects to the vendor listing page with a success message.
     *
     * @param UpdateVendor $request The incoming request containing the vendor data.
     *
     * @return \Illuminate\Http\RedirectResponse Redirect response to the vendor listing with a success message.
     */
    public function update(UpdateVendor $request)
    {
        $vendor = Vendor::find($request->id);
        $oldName = $vendor->name;
        $data = $request->only(["name"]);
        
        $vendor->update($data);
        //Remove all old category relations and add new based on request
        $this->service->createVendorCategoryRelation($vendor->id, $request->categories);

        if ($oldName == $request['name']) {
            return redirect(route('vendor-listing'))->with('message', 'Vendor updated successfully.');
        }
        $description = __('history.VendorUpdated', [
            'old_name' => $oldName,
            'new_name' => $request['name'],
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'vendor_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return redirect(route('vendor-listing'))->with('message', 'Vendor created successfully.');
    }

    /**
     * Retrieve vendor details based on the provided ID.
     *
     * This function fetches the vendor from the database using the given ID
     * and returns the vendor's ID and name in JSON format.
     *
     * @param int $id The ID of the vendor to retrieve.
     *
     * @return \Illuminate\Http\JsonResponse The vendor's ID and name in JSON format.
     */
    public function edit($id)
    {
        $vendor = Vendor::with('vendorCategories')->find($id);
        $data['id'] = $vendor->id;
        $data['name'] = $vendor->name;

        $data['selectedCategories'] = $vendor ? $vendor->vendorCategories->pluck('id')->toArray() : [];

        return response()->json($data);
    }

    /**
     * Function to remove the vendor
     * @param UpdateVendor $request
     *
     * @return [type]
     */
    public function deleteVendor()
    {
        $vendor = Vendor::findOrFail(request('id'));
        if ($vendor->count()) {

            if ($vendor->softwareAssets->count() > 0 || $vendor->assets->count() > 0) {
                return response()->json('This vendor is in use and cannot be deleted.');
            }

            $vendor->vendorCategories()->detach();
            $description = __('history.VendorDeleted', [
                'vendor' => $vendor->name,
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'vendor_deleted',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));

            return $vendor->delete();
        }

        return true;
    }

    /** Get the storage locations based on search
     * @return [type]
     */
    public function getFilterVendor($field = 'name')
    {
        $search = request('search');
        if (!empty($search)) {
            $res = Vendor::where("$field", 'like', '%' . $search . '%')->orderBy("$field")->get();
            $result = $res->mapWithKeys(function ($item, $key) use ($field) {
                return [
                    $key => [
                        'id' => $item['id'],
                        'text' => $item[$field],
                    ]
                ];
            });
            if (empty($result)) {
                $result[] = ['id' => '0', 'text' => 'No result found'];
            }

            return response()->json($result);
        }
    }
}
