<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\TechnicalSpecs;
use Exception;
use App\Models\MakeAndModel;
use App\Services\Settings\TechnicalSpecsService;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;

class TechnicalSpecsController extends Controller
{

    /**
     * Initialize the controller with the TechnicalSpecsService.
     *
     * @param \App\Services\TechnicalSpecsService $technicalSpecsService The service to manage technical specifications
     */
    public function __construct(protected TechnicalSpecsService $technicalSpecsService) {}

    /**
     * Display the index view for technical specifications.
     *
     * @return \Illuminate\View\View The view for the technical specifications index
     */
    public function index()
    {
        $specs = $this->technicalSpecsService->getAllTechnicalSpecs();
        $specs = $specs->paginate(config('pagination.per_page'));

        $status = $specs->map(
            function ($item) {
                return collect($item)->only(
                    [
                        'id',
                        'status'
                    ]
                )->all();
            }
        );

        $status = json_encode($status);

        $hardwares = MakeAndModel::with('manufacturer')->get();

        return view('settings.technical-specs.technical-specs', compact('specs', 'hardwares', 'status'));
    }


    /**
     * Search for technical specifications and return the results as JSON.
     *
     * This method handles the request to search for technical specifications. It sorts, filters, and paginates the
     * technical specifications based on the provided request parameters. The results are returned as JSON, including
     * the rendered views for the technical specifications data and pagination links.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the search results and rendered views
     */
    public function search(Request $request)
    {
        if ($request->ajax()) {
            $pageLength = request('page_legth') ?? config('pagination.per_page');

            $sortColumns = [
                '4' => 'asset_types.name',
                '5' => 'manufacturers.name',
                '6' => 'make_and_models.name',
                '7' => 'details',
                '11' => 'assets_count',
            ];

            $sortColumn = $sortColumns[request('sort_column')] ?? 'make_and_models.name';
            $sortDirection = request('sort_direction') ?? 'asc';

            $techSpecs = $this->technicalSpecsService->getAllTechnicalSpecs();

            $techSpecs = $this->technicalSpecsService->filterTechnicalSpecs($techSpecs);

            $specs = $techSpecs->orderBy($sortColumn, $sortDirection)->paginate($pageLength);

            // Create the statuses collection of the technical specs to show the tick mark on the table.
            $status = $specs->map(function ($item) {
                return collect($item)->only(['id', 'status'])->all();
            });

            $view['techspec'] = view('settings.technical-specs.technical-specs-data', compact('specs'))->render();
            $view['links'] = view('settings.technical-specs.technical-specs-pagination-links', compact('specs'))->render();
            $view['status'] = $status;

            return response()->json($view);
        }
    }

    /**
     * Store a new technical specification.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message
     */
    public function store()
    {
        $this->validate(request(), [
            'make_and_model_id' => 'required',
            'technical_spec' => 'required',
            'original_value' => 'required',
        ]);

        try {
            $makeModel = MakeAndModel::find(request('make_and_model_id'));

            TechnicalSpecs::create([
                'make_and_model_id' => request('make_and_model_id'),
                'details' => request('technical_spec'),
                'original_value' => request('original_value'),
            ]);

            $description = __('history.TechnicalSpecsCreated', [
                'assetType' => optional($makeModel->assetType)->name,
                'manufacturer' => optional($makeModel->manufacturer)->name,
                'make_model' => $makeModel->name,
                'technical_spec' => request('technical_spec'),
                'original_value' => request('original_value'),
            ]);

            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'technical_specs_created',
                'description' => $description,
            ];

            event(new BulkUpdates($assetHistory));

            return redirect('/technical-specs')->with('message', 'Technical specification created successfully.');
        } catch (Exception $e) {
            return redirect('/technical-specs')->with('error', 'Some error occured.');
        }
    }

    /**
     * Update an existing technical specification.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success of the update operation
     */
    public function update()
    {
        // Retrieve the old technical specification for history logging.
        $oldSpecs = TechnicalSpecs::with(['makeAndModel.manufacturer', 'makeAndModel.assetType'])->where('id', request('id'))->first();
        $oldTechinalSpecs = $oldSpecs->details;
        $oldSalvageValue = $oldSpecs->new_value;
        $oldOriginalValue = $oldSpecs->original_value;

        $newTechnicalSpecs = request('technical_spec');

        if (strpos(request('original_value'), 'USD') !== false) {
            $newOriginalValue = substr(request('original_value'), 0, -3);
        } else {
            $newOriginalValue = request('original_value');
        }

        $newSalvageValue = request('new_value');

        TechnicalSpecs::findOrFail(request('id'))
            ->update([
                'details' => request('technical_spec'),
                'original_value' => $newOriginalValue,
                'new_value' => $newSalvageValue
            ]);

        $description = '';

        if ($oldTechinalSpecs != $newTechnicalSpecs) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Tech Specs',
                'oldValue' => $oldTechinalSpecs,
                'newValue' => $newTechnicalSpecs,
            ]);
        }

        if ($oldSalvageValue != $newSalvageValue) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Savage Value',
                'oldValue' => $oldSalvageValue,
                'newValue' => $newSalvageValue,
            ]);
        }

        if ($oldOriginalValue != $newOriginalValue) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Original Value',
                'oldValue' => $oldOriginalValue,
                'newValue' => $newOriginalValue,
            ]);
        }

        $description = __('history.TechnicalSpecsUpdated', [
            'description' => $description,
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'technical_specs_updated',
            'description' => $description,
        ];

        event(new BulkUpdates($assetHistory));

        return response()->json('success');
    }


    /**
     * Update the status of a technical specification.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success of the update operation and the new status
     */
    public function updateStatus()
    {
        $technicalSpec = TechnicalSpecs::findOrFail(request('id'));

        $description = '';

        if ($technicalSpec->status == 0) {
            $technicalSpec->update(['status' => 1]);
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Status',
                'oldValue' => 'Inactive',
                'newValue' => 'Active',
            ]);
        } else {
            $technicalSpec->update(['status' => 0]);
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Status',
                'oldValue' => 'Active',
                'newValue' => 'Inactive',
            ]);
        }

        $description = __('history.TechnicalSpecsUpdated', [
            'description' => $description,
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'technical_specs_updated',
            'description' => $description,
        ];

        event(new BulkUpdates($assetHistory));

        $technicalSpec = TechnicalSpecs::findOrFail(request('id'));
        $status = $technicalSpec->status;

        $response = [
            'success' => 'success',
            'status' => $status,
            'message' => 'Status updated successfully.',
        ];

        return response()->json($response);
    }

    /**
     * Handle AJAX requests for various actions on technical specifications.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success or failure of the requested action
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        } elseif (request('action') == 'status') {
            return $this->updateStatus();
        }

        return response()->json(true);
    }

    /**
     * Export technical specifications to a CSV file.
     *
     * This method handles the request to export technical specifications to a CSV file. It retrieves, filters,
     * and processes the technical specifications in chunks, and prepares the CSV file for download.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The response containing the CSV file for download
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $specs = $this->technicalSpecsService->getAllTechnicalSpecs();
        $specs = $this->technicalSpecsService->filterTechnicalSpecs($specs);

        $specs->chunk(5000, function ($specsChunks) use ($data, $datas) {
            $datas->push($this->technicalSpecsService->getTechnicalSpecsExportData($specsChunks, 0, $data));
        });

        return exportToCsv($datas->toArray());
    }

    /**
     * Delete a technical specification.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success or failure of the delete operation
     */
    public function destroy()
    {
        if (!$this->deleteTechSpec(request('id'))) {
            return response()->json('error');
        }

        return response()->json("deleted");
    }

    /**
     * Delete a technical specification.
     *
     * @param int $technicalSpecId The ID of the technical specification to be deleted
     *
     * @return bool Returns true if the deletion was successful, false otherwise
     */
    private function deleteTechSpec($technicalSpecId)
    {
        $technicalSpec = TechnicalSpecs::findOrFail($technicalSpecId);

        if (optional($technicalSpec->assets)->count() > 0) {
            return false;
        }

        $technicalSpec->delete();

        $description = __('history.TechnicalSpecsDeleted', [
            'technicalSpecs' => $technicalSpec->details,
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'technical_specs_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));

        return true;
    }

    /**
     * Bulk delete technical specifications.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response indicating the success of the bulk delete operation
     */
    public function bulkDestroy()
    {
        $techicalSpecs = TechnicalSpecs::whereIn('id', request('tech_spec_ids'))->get();

        foreach ($techicalSpecs as $spec) {
            $this->deleteTechSpec($spec->id);
        }

        return response()->json("deleted");
    }
}
