<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\State;
use App\Models\AssetHistory;
use Illuminate\Support\Facades\Auth;
use App\Models\Country;
use Carbon\Carbon;

class StateController extends Controller
{
    public function index()
    {
        $states = State::with('country');
        $limit = 50;
        $num = num_row(request('page'), $limit);

        $countries = Country::orderBy('name')->pluck('name', 'name');

        if (request('searchText')) {
            $searchText = request('searchText');
            $states = $states->where('name', 'like', '%' . $searchText . '%');
        }
        $states = $states->orderBy('name')->paginate($limit);
        return view('settings.state.state', compact('states', 'countries', 'num'));
    }

    public function store()
    {
        $this->validate(request(), [
            'country_id' => 'required',
            'state' => 'required',
        ]);

        $state = request('state');
        $stateData = State::where('name', $state)->where('country_id', request('country_id'))->first();
        if (!$stateData) {
            State::create([
                'name' => $state,
                'country_id' => request('country_id')
            ]);

            $description = __('history.StateCreated', [
                'state' => $state
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'state_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($assetHistory));
            return redirect('/state');
        } else {
            return redirect('/state')->withErrors(['State already exists']);
        }
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    public function update()
    {
        $stateRow = State::findOrFail(request('state_id'));
        $oldState = $stateRow->name;
        $oldCountry = Country::find($stateRow->country_id) ? Country::find($stateRow->country_id)->name : '';

        $newState = request('name');
        $newCountry = request('country');
        $country = Country::where('name', $newCountry)->first();
        $stateRow->update(['name' => request('name'), 'country_id' => $country->id]);


        $description = '';
        if ($oldState != $newState) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'State',
                'oldValue'  => $oldState,
                'newValue'  => $newState,
            ]);
        }

        if ($oldCountry != $newCountry) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Country',
                'oldValue'  => $oldCountry,
                'newValue'  => $newCountry,
            ]);
        }

        $description = __('history.StateUpdated', [
            'description' => $description
        ]);

        $assetHistory = [
            'user_id'         => Auth::id(),
            'action'         => 'state_updated',
            'description'     => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
        return response()->json('success');
    }

    public function destroy()
    {
        $state = State::findOrFail(request('state_id'));
        $stateValue = $state->name;

        $state->delete();
        $description = __('history.StateDeleted', [
            'state' => $stateValue
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'state_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
        return response()->json("Deleted Successfully");
    }
}
